const pool = require("../config/db.js");

// Create a new batch
module.exports.createBatch = async (req, res) => {
  let connection;
  try {
    const {
      name,
      start_date,
      expected_end_date,
      status = "Active"
    } = req.body;

    // Validation
    if (!name || !start_date || !expected_end_date) {
      return res.status(400).json({
        status: 400,
        error: { msg: "Missing required fields: name, start_date, expected_end_date" }
      });
    }

    connection = await pool.getConnection();

    const [existingBatch] = await connection.query(
      "SELECT * FROM batches WHERE name = ? AND deleted_at is null",
      [name]
    );

    if (existingBatch.length > 0) {
      return res.status(409).json({
        status: 409,
        error: { msg: "A batch with this name already exists" }
      });
    }

    const [result] = await connection.query(
      `INSERT INTO batches (name, start_date, expected_end_date, status) 
       VALUES (?, ?, ?, ?)`,
      [name, start_date, expected_end_date, status]
    );

    return res.status(201).json({
      status: 201,
      success: { 
        msg: "Batch created successfully",
        batch_id: result.insertId
      }
    });
  } catch (error) {
    console.error("Error creating batch:", error);
    return res.status(500).json({
      status: 500,
      error: { msg: "Internal server error" }
    });
  } finally {
    if (connection) connection.release();
  }
};


// Get a batch by ID
module.exports.getBatchById = async (req, res) => {
  let connection;
  try {
    const { batch_id } = req.body;

    if (!batch_id) {
      return res.status(400).json({
        status: 400,
        error: { msg: "Batch ID is required" }
      });
    }

    connection = await pool.getConnection();
    
    const [batch] = await connection.query(
      "SELECT * FROM batches WHERE batch_id = ? AND deleted_at is null",
      [batch_id]
    );

    if (batch.length === 0) {
      return res.status(404).json({
        status: 404,
        error: { msg: "Batch not found" }
      });
    }

    return res.status(200).json({
      status: 200,
      batch: batch[0]
    });
  } catch (error) {
    console.error("Error fetching batch:", error);
    return res.status(500).json({
      status: 500,
      error: { msg: "Internal server error" }
    });
  } finally {
    if (connection) connection.release();
  }
};

// Get all batches with pagination and filtering
module.exports.getAllBatches = async (req, res) => {
  let connection;
 
  try {
    const {
      searchKeyWord = "",
      status,
      pageNumber = 1,
      pageSize = 10
    } = req.body;
    
    const offset = (pageNumber - 1) * pageSize;
    connection = await pool.getConnection();

    let whereClause = "WHERE 1=1";
    const params = [];

    if (searchKeyWord) {
      whereClause += " AND b.name LIKE ?";
      params.push(`%${searchKeyWord}%`);
    }

    if (status) {
      whereClause += " AND status = ?";
      params.push(status);
    }

    const [totalResult] = await connection.query(
      `SELECT COUNT(*) as total FROM batches b ${whereClause} AND b.deleted_at is null`,
      params
    );

    const countParams = [...params];
    params.push(parseInt(pageSize), offset);
    
    const [batches] = await connection.query(
      `SELECT b.*
       FROM batches b
       ${whereClause} AND b.deleted_at is null
       ORDER BY b.created_at DESC 
       LIMIT ? OFFSET ?`,
      params
    );
    
    return res.status(200).json({
      status: 200,
      total: totalResult[0].total,
      pageSize,
      pageNumber,
      totalPages: Math.ceil(totalResult[0].total / pageSize),
      batches
    });
  } catch (error) {
    console.error("Error fetching batches:", error);
    return res.status(500).json({
      status: 500,
      error: { msg: "Internal server error" }
    });
  } finally {
    if (connection) connection.release();
  }
};


// Update a batch
module.exports.updateBatch = async (req, res) => {
  let connection;
  try {
    const {
      batch_id,
      name,
      start_date,
      expected_end_date,
      status
    } = req.body;

    if (!batch_id) {
      return res.status(400).json({
        status: 400,
        error: { msg: "Batch ID is required" }
      });
    }

    connection = await pool.getConnection();

    const [existingBatch] = await connection.query(
      "SELECT * FROM batches WHERE batch_id = ? AND deleted_at is null",
      [batch_id]
    );

    if (existingBatch.length === 0) {
      return res.status(404).json({
        status: 404,
        error: { msg: "Batch not found" }
      });
    }

    if (name && name !== existingBatch[0].name) {
      const [nameExists] = await connection.query(
        "SELECT * FROM batches WHERE name = ? AND batch_id != ?",
        [name, batch_id]
      );

      if (nameExists.length > 0) {
        return res.status(409).json({
          status: 409,
          error: { msg: "Another batch with this name already exists" }
        });
      }
    }

    const updateFields = [];
    const updateParams = [];

    if (name) {
      updateFields.push("name = ?");
      updateParams.push(name);
    }

    if (start_date) {
      updateFields.push("start_date = ?");
      updateParams.push(start_date);
    }

    if (expected_end_date) {
      updateFields.push("expected_end_date = ?");
      updateParams.push(expected_end_date);
    }

    if (status) {
      updateFields.push("status = ?");
      updateParams.push(status);
    }

    if (updateFields.length === 0) {
      return res.status(400).json({
        status: 400,
        error: { msg: "No fields provided for update" }
      });
    }

    updateParams.push(batch_id);

    const [updateResult] = await connection.query(
      `UPDATE batches SET ${updateFields.join(", ")} WHERE batch_id = ?`,
      updateParams
    );

    if (updateResult.affectedRows === 0) {
      return res.status(500).json({
        status: 500,
        error: { msg: "Failed to update batch" }
      });
    }

    return res.status(200).json({
      status: 200,
      success: { msg: "Batch updated successfully" }
    });
  } catch (error) {
    console.error("Error updating batch:", error);
    return res.status(500).json({
      status: 500,
      error: { msg: "Internal server error" }
    });
  } finally {
    if (connection) connection.release();
  }
};


// Delete a batch (with check for associated students)
module.exports.deleteBatch = async (req, res) => {
  try {
    const { ids } = req.body; // expecting array of batch_ids

    if (!Array.isArray(ids) || ids.length === 0) {
      return res.status(400).json({
        status: 400,
        error: { msg: "An array of batch IDs is required" }
      });
    }

    const con = await pool.getConnection();
    try {
      // Check if batches exist and are not already deleted
      const [existingBatches] = await con.query(
        `SELECT * FROM batches WHERE batch_id IN (?) AND deleted_at IS NULL`,
        [ids]
      );

      if (existingBatches.length === 0) {
        return res.status(404).json({
          status: 404,
          error: { msg: "Batches not found" }
        });
      }

      // Check if any batch has students
      const [studentCounts] = await con.query(
        `SELECT batch_id, COUNT(*) as count FROM batch_student WHERE batch_id IN (?) GROUP BY batch_id`,
        [ids]
      );

      if (studentCounts.length > 0) {
        return res.status(400).json({
          status: 400,
          error: {
            msg: "Cannot delete batches with associated students. Remove students from batches first.",
            batches_with_students: studentCounts
          }
        });
      }

      // Soft delete batches
      const [deleteResult] = await con.query(
        `UPDATE batches SET deleted_at = NOW() WHERE batch_id IN (?)`,
        [ids]
      );

      if (deleteResult.affectedRows > 0) {
        return res.status(200).json({
          status: 200,
          msg: "Batches deleted successfully"
        });
      } else {
        return res.status(404).json({
          status: 404,
          error: { msg: "No matching batches found to delete" }
        });
      }
    } catch (error) {
      console.error("Error:", error);
      return res.status(500).json({ error: { msg: "Internal server error" } });
    } finally {
      con.release();
    }
  } catch (error) {
    console.error("Database connection error:", error);
    return res.status(500).json({ error: { msg: "Internal server error" } });
  }
};
