const pool = require("../config/db.js");

// Create (Assign) students to a batch
module.exports.createBatchStudent = async (req, res) => {
  let connection;
  try {
    const {
      class_id,
      section_id,
      session_id,
      batch_id,
      degree_id,
      student_ids,
      enrollment_date = new Date().toISOString().split('T')[0],
      status_in_batch = 'Active',
      education_type_id,
      is_current = 1,
    } = req.body;

    // Validate required fields
    if (
      !student_ids ||
      !Array.isArray(student_ids) ||
      student_ids.length === 0 ||
      !education_type_id ||
      (parseInt(education_type_id, 10) === 2 && (!batch_id || !degree_id)) ||
      (parseInt(education_type_id, 10) === 1 && (!class_id || !section_id || !session_id))
    ) {
      return res.status(400).json({
        status: 400,
        error: {
          msg: "Missing required fields: student_ids, education_type_id, and relevant identifiers (class_id, section_id, session_id OR batch_id, degree_id)",
        },
      });
    }

    connection = await pool.getConnection();
    await connection.beginTransaction();

    // Check all students exist
    const [studentsData] = await connection.query(
      "SELECT id FROM students WHERE id IN (?)",
      [student_ids]
    );
    const existingStudentIds = studentsData.map((s) => s.id.toString());

    if (existingStudentIds.length !== student_ids.length) {
      await connection.rollback();
      const notFoundStudentIds = student_ids.filter(
        (id) => !existingStudentIds.includes(id.toString())
      );
      return res.status(404).json({
        status: 404,
        error: {
          msg: `One or more students not found. Missing IDs: ${notFoundStudentIds.join(", ")}`,
        },
      });
    }

    const operationSummary = {
      created_count: 0,
      already_exists_in_batch_count: 0,
      failed_already_current_count: 0,
      created_entries: [],
      already_exists_in_batch_entries: [],
      failed_already_current_entries: [],
    };

    for (const student_id of student_ids) {
      let existsQuery = '';
      let existsParams = [];

      if (parseInt(education_type_id, 10) === 1) {
        existsQuery = `
          SELECT batch_student_id FROM batch_student 
          WHERE class_id = ? AND section_id = ? AND session_id = ? AND student_id = ?
        `;
        existsParams = [class_id, section_id, session_id, student_id];
      } else {
        existsQuery = `
          SELECT batch_student_id FROM batch_student 
          WHERE batch_id = ? AND student_id = ?
        `;
        existsParams = [batch_id, student_id];
      }

      const [exists] = await connection.query(existsQuery, existsParams);

      if (exists.length > 0) {
        operationSummary.already_exists_in_batch_count++;
        operationSummary.already_exists_in_batch_entries.push({ student_id });
        continue;
      }

      // Check is_current constraint
      if (is_current === 1 || is_current === true) {
        const [isCurrentElsewhere] = await connection.query(
          `SELECT batch_student_id FROM batch_student WHERE student_id = ? AND is_current = 1`,
          [student_id]
        );

        if (isCurrentElsewhere.length > 0) {
          operationSummary.failed_already_current_count++;
          operationSummary.failed_already_current_entries.push({ student_id });
          continue;
        }
      }

      // Insert new entry
      let insertQuery = '';
      let insertParams = [];

      if (parseInt(education_type_id, 10) === 1) {
        insertQuery = `
          INSERT INTO batch_student (class_id, section_id, session_id, student_id, enrollment_date, status_in_batch, edu_typ_id, is_current)
          VALUES (?, ?, ?, ?, ?, ?, ?, ?)
        `;
        insertParams = [class_id, section_id, session_id, student_id, enrollment_date, status_in_batch, education_type_id, is_current];
      } else {
        insertQuery = `
          INSERT INTO batch_student (batch_id, degree_id, student_id, enrollment_date, status_in_batch, edu_typ_id, is_current)
          VALUES (?, ?, ?, ?, ?, ?, ?)
        `;
        insertParams = [batch_id, degree_id, student_id, enrollment_date, status_in_batch, education_type_id, is_current];
      }

      const [result] = await connection.query(insertQuery, insertParams);
      operationSummary.created_count++;
      operationSummary.created_entries.push({
        student_id,
        batch_student_id: result.insertId,
      });
    }

    await connection.commit();

    return res.status(201).json({
      status: 201,
      success: {
        msg: "Batch student assignments processed successfully.",
        summary: operationSummary,
      },
    });
  } catch (error) {
    console.error("Failed to process batch student assignment:", error);
    if (connection) await connection.rollback();
    return res.status(500).json({
      status: 500,
      error: { msg: "Internal server error." },
    });
  } finally {
    if (connection) connection.release();
  }
};



// Get a specific batch_student entry by its ID


module.exports.getBatchStudentById = async (req, res) => {
  let connection;
  try {
    const { batch_student_id } = req.body;
   

    if (!batch_student_id) {
      return res.status(400).json({ status: 400, error: { msg: "student_id is required" } });
    }

    connection = await pool.getConnection();

    const query = `
      SELECT 
          bs.batch_student_id,
          bs.student_id,
          bs.enrollment_date,
          bs.status_in_batch,
          bs.edu_typ_id,
          s.first_name as student_first_name,
          s.last_name as student_last_name,
          s.email as student_email,
          s.institute_reg_num as student_registration,
          bs.batch_id,
          b.name as batch_name,
          b.degree_id,
          deg.name as degree_name,
          bs.class_section_id,
          cs.class_id,
          c.class_name,
          cs.session_id,
          ss.session_name,
          bs.created_at,
          bs.updated_at
       FROM batch_student bs
       JOIN students s ON bs.student_id = s.id
      LEFT JOIN batches b ON bs.batch_id = b.batch_id
       LEFT JOIN degrees deg ON b.degree_id = deg.degree_id
       LEFT JOIN class_sections cs ON bs.class_section_id = cs.class_section_id
       LEFT JOIN classes c ON cs.class_id = c.class_id
       LEFT JOIN sessions ss ON cs.session_id = ss.session_id
       WHERE bs.batch_student_id = ?
    `;

    const [results] = await connection.query(query, [batch_student_id]);

    if (results.length === 0) {
      return res.status(404).json({ status: 404, error: { msg: "Batch student entry not found" } });
    }

   
    const entry = results[0];

    return res.status(200).json({ status: 200, batch_student: entry });

  } catch (error) {
    console.error("Error fetching batch student entry:", error);
    return res.status(500).json({ status: 500, error: { msg: "Internal server error" } });
  } finally {
    if (connection) connection.release();
  }
};
// Get All Batch Student Entries (with filters and pagination)


module.exports.getAllBatchStudents = async (req, res) => {
  console.log(req.body);
  let connection;
  try {
    const {
      searchKeyWord = "",
      student_id,
      status_in_batch,
      education_type_id,
      is_current = 1, 
      class_ids,
      session_ids,
      batch_ids,
      degree_ids,
      section_ids,
      pageNumber = 1,
      pageSize = 100
    } = req.body;

    if (!education_type_id) {
      return res.status(400).json({ status: 400, error: { msg: "education_type_id is required." } });
    }

    const offset = (parseInt(pageNumber, 10) - 1) * parseInt(pageSize, 10);
    connection = await pool.getConnection();

    let whereClauses = [];
    const params = [];

    let selectFields = '';
    let joinClause = '';

    if (parseInt(education_type_id, 10) === 1) {
      // School: use class_id, section_id, session_id directly from batch_student
      selectFields = `
        bs.class_id,
        bs.section_id,
        bs.session_id,
        c.class_name AS degree_name,
        ss.session_name AS batch_name
      `;
      joinClause = `
        LEFT JOIN classes c ON bs.class_id = c.class_id
        LEFT JOIN sessions ss ON bs.session_id = ss.session_id
      `;

      if (class_ids?.length) {
        whereClauses.push("bs.class_id IN (?)");
        params.push(class_ids);
      }
      if (section_ids?.length) {
        whereClauses.push("bs.section_id IN (?)");
        params.push(section_ids);
      }
      if (session_ids?.length) {
        whereClauses.push("bs.session_id IN (?)");
        params.push(session_ids);
      }

    } else if (parseInt(education_type_id, 10) === 2) {
      // College
      selectFields = `
        bs.batch_id,
        b.name AS batch_name,
        bs.degree_id,
        deg.name AS degree_name
      `;
      joinClause = `
        JOIN batches b ON bs.batch_id = b.batch_id
        LEFT JOIN degrees deg ON bs.degree_id = deg.degree_id
      `;

      if (batch_ids?.length) {
        whereClauses.push("bs.batch_id IN (?)");
        params.push(batch_ids);
      }
      if (degree_ids?.length) {
        whereClauses.push("bs.degree_id IN (?)");
        params.push(degree_ids);
      }

    } else {
      return res.status(400).json({ status: 400, error: { msg: "Invalid education_type_id." } });
    }

    whereClauses.push("bs.edu_typ_id = ?");
    params.push(education_type_id);

    if (student_id) {
      whereClauses.push("bs.student_id = ?");
      params.push(student_id);
    }

    if (status_in_batch) {
      whereClauses.push("bs.status_in_batch = ?");
      params.push(status_in_batch);
    }

    if (is_current === 0 || is_current === 1) {
      whereClauses.push("bs.is_current = ?");
      params.push(is_current);
    }

    if (searchKeyWord) {
      const pattern = `%${searchKeyWord}%`;
      whereClauses.push(`
        (s.first_name LIKE ? OR s.last_name LIKE ? OR s.institute_reg_num LIKE ?)
      `);
      params.push(pattern, pattern, pattern);
    }

    const whereString = whereClauses.length > 0 ? `WHERE ${whereClauses.join(" AND ")}` : "";

    const countQuery = `
      SELECT COUNT(bs.batch_student_id) AS total
      FROM batch_student bs
      JOIN students s ON bs.student_id = s.id
      ${joinClause}
      ${whereString}
    `;

    const [totalResult] = await connection.query(countQuery, params);
    const totalRecords = totalResult[0]?.total || 0;

    const queryParams = [...params, parseInt(pageSize, 10), offset];
    const dataQuery = `
      SELECT 
        bs.batch_student_id,
        ${selectFields},
        bs.student_id,
        s.institute_reg_num AS student_registration,
        s.first_name AS student_first_name,
        s.last_name AS student_last_name,
        bs.enrollment_date,
        bs.status_in_batch,
        bs.is_current,
        bs.created_at,
        bs.updated_at
      FROM batch_student bs
      JOIN students s ON bs.student_id = s.id
      ${joinClause}
      ${whereString}
      ORDER BY bs.created_at DESC
      LIMIT ? OFFSET ?
    `;

    const [entries] = await connection.query(dataQuery, queryParams);

    return res.status(200).json({
      status: 200,
      total: totalRecords,
      pageSize: parseInt(pageSize, 10),
      pageNumber: parseInt(pageNumber, 10),
      totalPages: Math.ceil(totalRecords / parseInt(pageSize, 10)),
      batch_students: entries
    });

  } catch (error) {
    console.error("Error fetching all batch student entries:", error);
    return res.status(500).json({
      status: 500,
      error: { msg: "Internal server error" }
    });
  } finally {
    if (connection) connection.release();
  }
};

// Update a batch_student entry (e.g., status_in_batch, enrollment_date)
module.exports.updateBatchStudent = async (req, res) => {
  let connection;
  console.log(req.body);
  try {
    const { batch_student_id, enrollment_date, status_in_batch, reason } = req.body;

    if (!batch_student_id) {
      return res.status(400).json({ status: 400, error: { msg: "student_id is required" } });
    }
    if (!enrollment_date && !status_in_batch) {
      return res.status(400).json({ status: 400, error: { msg: "At least one field (enrollment date, student status) must be provided for update" } });
    }
    if (status_in_batch && !reason) {
        return res.status(400).json({ status: 400, error: { msg: "Reason is required when updating student status" } });
    }

    connection = await pool.getConnection();
    await connection.beginTransaction();

    try {
        // Fetch current state if status is being updated, for history
        let previousStatus = null;
        if (status_in_batch) {
            const [currentEntry] = await connection.query(
                "SELECT status_in_batch FROM batch_student WHERE batch_student_id = ?",
                [batch_student_id]
            );
            if (currentEntry.length === 0) {
                await connection.rollback();
                return res.status(404).json({ status: 404, error: { msg: "student entry not found" } });
            }
            // Only log history if the status is actually changing
            if (currentEntry[0].status_in_batch !== status_in_batch) {
                 previousStatus = currentEntry[0].status_in_batch;
            } else if (currentEntry[0].status_in_batch === status_in_batch && !enrollment_date ) {
                // If status is the same and no other field is changing, no need to proceed or log
                await connection.rollback();
                return res.status(200).json({ status: 200, success: { msg: "No changes detected for student entry." }});
            }
        }

        const updateFields = [];
        const updateParams = [];

        if (enrollment_date) {
            updateFields.push("enrollment_date = ?");
            updateParams.push(enrollment_date);
        }
        if (status_in_batch) {
            updateFields.push("status_in_batch = ?");
            updateParams.push(status_in_batch);
        }
        
        
        if (updateFields.length === 0) {
             await connection.rollback(); // Should not happen due to prior checks, but as a safeguard
             return res.status(400).json({ status: 400, error: { msg: "No valid fields to update." }});
        }
        updateParams.push(batch_student_id); // For the WHERE clause

        const [updateResult] = await connection.query(
            `UPDATE batch_student SET ${updateFields.join(", ")} WHERE batch_student_id = ?`,
            updateParams
        );

        if (updateResult.affectedRows === 0 && previousStatus === null && (enrollment_date)) {
             // This case means only enrollment_date was updated, but the record wasn't found
             await connection.rollback();
             return res.status(404).json({
                 status: 404,
                 error: { msg: "Student  entry not found" }
             });
        }
        // If only enrollment_date was changed, and it was successful, previousStatus would be null.
        // If status_in_batch was changed (previousStatus is not null), log it.
        if (previousStatus !== null && status_in_batch) { // Ensure status_in_batch was part of the update request
            await connection.query(
                "INSERT INTO batch_student_status_history (batch_student_id, previous_status, new_status, reason) VALUES (?, ?, ?, ?)",
                [batch_student_id, previousStatus, status_in_batch, reason]
            );
        }

        await connection.commit();
        return res.status(200).json({
            status: 200,
            success: { msg: "Student  updated successfully" }
        });

    } catch (error) {
        await connection.rollback();
        console.error("Error during batch student update transaction:", error);
        return res.status(500).json({ status: 500, error: { msg: "Internal server error during transaction" } });
    }
  } catch (error) {
    console.error("Error updating batch student entry:", error);
    return res.status(500).json({ status: 500, error: { msg: "Internal server error" } });
  } finally {
    if (connection) connection.release();
  }
};

// Delete a batch_student entry (Remove student from a batch)
// This can be by batch_student_id or by a combination of batch_id and student_id(s)
module.exports.deleteBatchStudent = async (req, res) => {
  let connection;
  try {
    const { batch_student_id, batch_id, student_ids } = req.body;

    if (!batch_student_id && (!batch_id || !student_ids || !Array.isArray(student_ids) || student_ids.length === 0)) {
      return res.status(400).json({
        status: 400,
        error: { msg: "Either batch_student_id or (batch_id AND student_ids array) is required" }
      });
    }

    connection = await pool.getConnection();
    let deleteResult;

    if (batch_student_id) {
      [deleteResult] = await connection.query(
        "DELETE FROM batch_student WHERE batch_student_id = ?",
        [batch_student_id]
      );
    } else {
      // Ensure student_ids is an array, even if a single ID string is passed for robustness
      const idsArray = Array.isArray(student_ids) ? student_ids : [student_ids];
      if (idsArray.length === 0) {
        return res.status(400).json({status: 400, error: { msg: "student_ids array cannot be empty when deleting by batch_id" }});
      }
      [deleteResult] = await connection.query(
        "DELETE FROM batch_student WHERE batch_id = ? AND student_id IN (?)",
        [batch_id, idsArray]
      );
    }

    if (deleteResult.affectedRows === 0) {
        return res.status(404).json({
            status: 404,
            error: { msg: "No matching batch student entries found to delete." }
        });
    }

    return res.status(200).json({
      status: 200,
      success: { 
        msg: "Batch student entries deleted successfully",
        deleted_count: deleteResult.affectedRows
      }
    });
  } catch (error) {
    console.error("Error deleting batch student entries:", error);
    return res.status(500).json({ status: 500, error: { msg: "Internal server error" } });
  } finally {
    if (connection) connection.release();
  }
};

// This function remains to get students for a specific batch (as per original /list route)
// It's distinct from getAllBatchStudents which lists all batch-student links globally.
module.exports.getStudentsByBatchId = async (req, res) => {
  let connection;
  try {
    const { 
      batch_id, 
      searchKeyWord = "",
      status_in_batch,
      pageNumber = 1, 
      pageSize = 10 
    } = req.body;
    
    if (!batch_id) {
      return res.status(400).json({ status: 400, error: { msg: "Batch ID is required" } });
    }

    const offset = (parseInt(pageNumber, 10) - 1) * parseInt(pageSize, 10);
    connection = await pool.getConnection();

    let whereClause = "WHERE bs.batch_id = ?";
    const params = [batch_id];

    if (status_in_batch) {
      whereClause += " AND bs.status_in_batch = ?";
      params.push(status_in_batch);
    }

    if (searchKeyWord) {
      whereClause += ` AND (
        s.first_name LIKE ? OR 
        s.last_name LIKE ? OR 
        s.email LIKE ? OR 
        s.registration_number LIKE ? OR 
        s.phone_number LIKE ?
      )`;
      const searchPattern = `%${searchKeyWord}%`;
      params.push(searchPattern, searchPattern, searchPattern, searchPattern, searchPattern);
    }

    const [totalResult] = await connection.query(
      `SELECT COUNT(*) as total 
       FROM batch_student bs
       JOIN students s ON bs.student_id = s.id
       ${whereClause}`,
      params
    );
    const totalRecords = totalResult[0].total;

    const queryParams = [...params, parseInt(pageSize, 10), offset];
    const [students] = await connection.query(
      `SELECT 
        s.id as student_id,
        s.first_name,
        s.last_name,
        s.email,
        s.phone_number,
        s.registration_number,
        s.status as student_general_status,
        bs.batch_student_id,
        bs.enrollment_date,
        bs.status_in_batch,
        bs.created_at as assignment_created_at,
        bs.updated_at as assignment_updated_at
       FROM batch_student bs
       JOIN students s ON bs.student_id = s.id
       ${whereClause}
       ORDER BY s.first_name ASC, s.last_name ASC
       LIMIT ? OFFSET ?`,
      queryParams
    );

    return res.status(200).json({
      status: 200,
      total: totalRecords,
      pageSize: parseInt(pageSize, 10),
      pageNumber: parseInt(pageNumber, 10),
      totalPages: Math.ceil(totalRecords / parseInt(pageSize, 10)),
      students_in_batch: students // Renamed for clarity
    });
  } catch (error) {
    console.error("Error fetching students in specific batch:", error);
    return res.status(500).json({ status: 500, error: { msg: "Internal server error" } });
  } finally {
    if (connection) connection.release();
  }
};

// Get status change history for a batch_student entry
module.exports.getBatchStudentStatusHistory = async (req, res) => {
  let connection;
  try {
    const { batch_student_id, pageNumber = 1, pageSize = 10 } = req.body;

    if (!batch_student_id) {
      return res.status(400).json({ status: 400, error: { msg: "batch_student_id is required" } });
    }

    const offset = (parseInt(pageNumber, 10) - 1) * parseInt(pageSize, 10);
    connection = await pool.getConnection();

    const [studentBatchEntry] = await connection.query(
      "SELECT batch_student_id FROM batch_student WHERE batch_student_id = ?",
      [batch_student_id]
    );
    if (studentBatchEntry.length === 0) {
        return res.status(404).json({ status: 404, error: { msg: "Batch student entry not found." } });
    }

    // Get total count of history records for pagination
    const [totalResult] = await connection.query(
      "SELECT COUNT(*) as total FROM batch_student_status_history WHERE batch_student_id = ?",
      [batch_student_id]
    );
    const totalRecords = totalResult[0].total;

    // Fetch paginated history records
    const [historyRecords] = await connection.query(
      `SELECT status_history_id, batch_student_id, previous_status, new_status, reason, changed_at 
       FROM batch_student_status_history 
       WHERE batch_student_id = ? 
       ORDER BY changed_at DESC 
       LIMIT ? OFFSET ?`,
      [batch_student_id, parseInt(pageSize, 10), offset]
    );

    return res.status(200).json({
      status: 200,
      batch_student_id,
      total: totalRecords,
      pageSize: parseInt(pageSize, 10),
      pageNumber: parseInt(pageNumber, 10),
      totalPages: Math.ceil(totalRecords / parseInt(pageSize, 10)),
      history: historyRecords
    });

  } catch (error) {
    console.error("Error fetching batch student status history:", error);
    return res.status(500).json({ status: 500, error: { msg: "Internal server error" } });
  } finally {
    if (connection) connection.release();
  }
}; 

module.exports.getAllClassSection = async (req, res) => {
  try {
      const con = await pool.getConnection();

      try {
          // Fetch active session_id
          const [activeSessionRows] = await con.query("SELECT session_id FROM sessions WHERE is_active = 1 LIMIT 1");
          let activeSessionId = null;
          if (activeSessionRows.length > 0) {
              activeSessionId = activeSessionRows[0].session_id;
          }
          
          const [classSections] = await con.query(
              "SELECT cs.*, c.class_name, s.section_name " +
              "FROM class_sections cs " +
              "LEFT JOIN classes c ON cs.class_id = c.class_id " +
              "LEFT JOIN section s ON cs.section_id = s.section_id " +
              "WHERE cs.session_id = ? " +
              " ORDER BY s.section_name",
              [activeSessionId]
          );
              return res.status(200).json({ status: 200, classSections });    

      } catch (error) {
          console.error("Error:", error);
          return res.status(500).json({ error: { msg: "Internal server error" } });
      } finally {
          con.release();
      }
  } catch (error) {
      console.error("Database connection error:", error);
      return res.status(500).json({ error: { msg: "Internal server error" } });
  }
};

/**
 * Fetches a single student's main details along with their complete
 * academic history (all past and current enrollments).
 */
module.exports.getStudentAcademicHistory = async (req, res) => {
  let connection;
  try {
    const { student_id } = req.params; // Get the ID from the URL path

    if (!student_id) {
      return res.status(400).json({ status: 400, error: { msg: "Student ID is required." } });
    }

    connection = await pool.getConnection();

    // STEP 1: Fetch the student's primary details.
    const studentQuery = `
      SELECT id, institute_reg_num, first_name, last_name, email, phone 
      FROM students 
      WHERE id = ?
    `;
    const [studentResult] = await connection.query(studentQuery, [student_id]);

    if (studentResult.length === 0) {
      return res.status(404).json({ status: 404, error: { msg: "Student not found." } });
    }

    const studentDetails = studentResult[0];

    // STEP 2: Fetch all academic records for that student.
    // This query uses LEFT JOINs and CASE statements to handle both education types in one go.
    const historyQuery = `
      SELECT 
        bs.batch_student_id,
        bs.enrollment_date,
        bs.status_in_batch,
        bs.is_current,
        bs.edu_typ_id,
        CASE 
          WHEN bs.edu_typ_id = 1 THEN c.class_name
          WHEN bs.edu_typ_id = 2 THEN deg.name
          ELSE NULL 
        END AS program_name,
        CASE 
          WHEN bs.edu_typ_id = 1 THEN ss.session_name
          WHEN bs.edu_typ_id = 2 THEN b.name
          ELSE NULL 
        END AS batch_or_session_name
      FROM batch_student bs
      LEFT JOIN students s ON bs.student_id = s.id
      -- Joins for Education Type 1 (School)
      LEFT JOIN class_sections cs ON bs.class_section_id = cs.class_section_id AND bs.edu_typ_id = 1
      LEFT JOIN classes c ON cs.class_id = c.class_id
      LEFT JOIN sessions ss ON cs.session_id = ss.session_id
      -- Joins for Education Type 2 (College)
      LEFT JOIN batches b ON bs.batch_id = b.batch_id AND bs.edu_typ_id = 2
      LEFT JOIN degrees deg ON bs.degree_id = deg.degree_id
      WHERE bs.student_id = ?
      ORDER BY bs.enrollment_date DESC
    `;

    const [academic_records] = await connection.query(historyQuery, [student_id]);

    // STEP 3: Combine and send the final, structured response.
    return res.status(200).json({
      status: 200,
      student: studentDetails,
      academic_records: academic_records
    });

  } catch (error) {
    console.error("Error fetching student academic history:", error);
    return res.status(500).json({
      status: 500,
      error: { msg: "Internal server error" }
    });
  } finally {
    if (connection) connection.release();
  }
};

//attendance

module.exports.loadAttendance = async (req, res) => {
  console.log(req.body);
  console.log("loadAttendance");
  const connection = await pool.getConnection();
  try {
    const { class_id, section_id, session_id, attendance_date } = req.body;

    if (!class_id || !section_id || !session_id || !attendance_date) {
      return res.status(400).json({ status: 400, error: { msg: "class_id, section_id, session_id, and attendance_date are required." } });
    }
    console.log("attendance_date", attendance_date);
    // Ensure attendance_date is in correct format (YYYY-MM-DD)
    const formattedDate = new Date(attendance_date).toISOString().slice(0, 10);

    // 1. Create attendance_dates entry if not exists
    let [existingDate] = await connection.query(`
      SELECT id FROM attendance_dates 
      WHERE class_id = ? AND section_id = ? AND session_id = ? AND attendance_date = ?
    `, [class_id, section_id, session_id, formattedDate]);

    let attendance_date_id;
    if (existingDate.length > 0) {
      attendance_date_id = existingDate[0].id;
    } else {
      const [insertResult] = await connection.query(`
        INSERT INTO attendance_dates (class_id, section_id, session_id, attendance_date)
        VALUES (?, ?, ?, ?)
      `, [class_id, section_id, session_id, formattedDate]);

      attendance_date_id = insertResult.insertId;
    }

    // 2. Fetch students from batch_student
    const [students] = await connection.query(`
      SELECT 
        bs.student_id,
        s.first_name,
        s.last_name,
        s.institute_reg_num,
        ar.status,
        ar.remarks
      FROM batch_student bs
      JOIN students s ON bs.student_id = s.id
      LEFT JOIN attendance_records ar 
        ON ar.attendance_date_id = ? AND ar.student_id = bs.student_id
      WHERE bs.class_id = ? AND bs.section_id = ? AND bs.session_id = ? AND bs.is_current = 1
    `, [attendance_date_id, class_id, section_id, session_id]);
    return res.status(200).json({
      status: 200,
      attendance_date_id,
      students
    });

  } catch (error) {
    console.error("Error loading attendance:", error);
    return res.status(500).json({ status: 500, error: { msg: "Internal server error" } });
  } finally {
    if (connection) connection.release();
  }
};
module.exports.updateAttendance = async (req, res) => {
  const connection = await pool.getConnection();
  try {
    const { attendance_date_id, students } = req.body;

    if (!attendance_date_id || !Array.isArray(students) || students.length === 0) {
      return res.status(400).json({
        status: 400,
        error: { msg: "attendance_date_id and students array are required." }
      });
    }

    // Begin transaction
    await connection.beginTransaction();

    for (const student of students) {
      const { student_id, status, remarks } = student;

      if (!student_id || !status) {
        continue; // Skip invalid entries
      }

      // Try to update existing record
      const [existing] = await connection.query(`
        SELECT id FROM attendance_records
        WHERE attendance_date_id = ? AND student_id = ?
      `, [attendance_date_id, student_id]);

      if (existing.length > 0) {
        // Update existing
        await connection.query(`
          UPDATE attendance_records
          SET status = ?, remarks = ?, recorded_at = CURRENT_TIMESTAMP
          WHERE attendance_date_id = ? AND student_id = ?
        `, [status, remarks, attendance_date_id, student_id]);
      } else {
        // Insert new
        await connection.query(`
          INSERT INTO attendance_records (attendance_date_id, student_id, status, remarks)
          VALUES (?, ?, ?, ?)
        `, [attendance_date_id, student_id, status, remarks]);
      }
    }

    // Commit transaction
    await connection.commit();

    return res.status(200).json({
      status: 200,
      msg: "Attendance successfully updated."
    });

  } catch (error) {
    await connection.rollback();
    console.error("Error updating attendance:", error);
    return res.status(500).json({
      status: 500,
      error: { msg: "Internal server error" }
    });
  } finally {
    if (connection) connection.release();
  }
};
