const pool = require("../config/db.js");

module.exports.createDegree = async (req, res) => {
  try {
    const { name, code, description = "" } = req.body;
    const con = await pool.getConnection();
    console.log(req.body);

    try {
      // Check if degree code already exists (because code must be unique)
      const [degreeExists] = await con.query(
        `SELECT * FROM degrees WHERE code = ?`,
        [code]
      );
      if (degreeExists.length > 0) {
        return res
          .status(400)
          .json({
            status: 400,
            error: { msg: "Degree with this code already exists!" },
          });
      }

      // Insert new degree
      await con.query(
        `INSERT INTO degrees (name, code, description) VALUES (?, ?, ?)`,
        [name, code, description]
      );

      return res
        .status(200)
        .json({ status: 200, success: { msg: "Degree created successfully" } });
    } catch (error) {
      console.error("Error:", error);
      return res.status(500).json({ error: { msg: "Internal server error" } });
    } finally {
      con.release();
    }
  } catch (error) {
    console.error("Database connection error:", error);
    return res.status(500).json({ error: { msg: "Internal server error" } });
  }
};
module.exports.updateDegree = async (req, res) => {
  try {
    const { id, name, code, description = "" } = req.body;
    const con = await pool.getConnection();

    try {
      // Check if another degree has the same code
      const [degreeExists] = await con.query(
        `SELECT * FROM degrees WHERE code = ? AND degree_id != ?`,
        [code, id]
      );
      if (degreeExists.length > 0) {
        return res
          .status(400)
          .json({
            error: { msg: "Another degree with this code already exists!" },
          });
      }

      await con.query(
        `UPDATE degrees SET name = ?, code = ?, description = ? WHERE degree_id = ?`,
        [name, code, description, id]
      );

      return res
        .status(200)
        .json({ status: 200, success: { msg: "Degree updated successfully" } });
    } catch (error) {
      console.error("Error:", error);
      return res.status(500).json({ error: { msg: "Internal server error" } });
    } finally {
      con.release();
    }
  } catch (error) {
    console.error("Database connection error:", error);
    return res.status(500).json({ error: { msg: "Internal server error" } });
  }
};
module.exports.getAllDegree = async (req, res) => {
  try {
    const { searchKeyWord = "", pageNumber = 1, pageSize = 10 } = req.body;
    const offset = (pageNumber - 1) * pageSize;
    const con = await pool.getConnection();

    try {
      const [totalResult] = await con.query(
        `SELECT COUNT(*) AS total FROM degrees WHERE name LIKE ? AND is_deleted = FALSE`,
        [`%${searchKeyWord}%`]
      );

      const [degrees] = await con.query(
        `SELECT * FROM degrees WHERE name LIKE ? AND is_deleted = FALSE ORDER BY degree_id DESC LIMIT ? OFFSET ?`,
        [`%${searchKeyWord}%`, parseInt(pageSize), offset]
      );

      return res.status(200).json({
        status: 200,
        total: totalResult[0].total,
        pageSize,
        pageNumber,
        totalPages: Math.ceil(totalResult[0].total / pageSize),
        degrees,
      });
    } catch (error) {
      console.error("Error:", error);
      return res.status(500).json({ error: { msg: "Internal server error" } });
    } finally {
      con.release();
    }
  } catch (error) {
    console.error("Database connection error:", error);
    return res.status(500).json({ error: { msg: "Internal server error" } });
  }
};
module.exports.getDegreeById = async (req, res) => {
  try {
    const { id } = req.body;
    const con = await pool.getConnection();

    try {
      const [result] = await con.query(
        `SELECT * FROM degrees WHERE degree_id = ? AND is_deleted = FALSE`,
        [id]
      );

      if (result.length === 0) {
        return res
          .status(404)
          .json({ status: 404, error: { msg: "Degree not found" } });
      }

      return res.status(200).json({ status: 200, degree: result[0] });
    } catch (error) {
      console.error("Error:", error);
      return res.status(500).json({ error: { msg: "Internal server error" } });
    } finally {
      con.release();
    }
  } catch (error) {
    console.error("Database connection error:", error);
    return res.status(500).json({ error: { msg: "Internal server error" } });
  }
};
module.exports.deleteDegree = async (req, res) => {
  try {
    const { ids } = req.body;
    const con = await pool.getConnection();
    try {
      const [degreeExists] = await con.query(
        `SELECT * FROM degrees WHERE degree_id IN (?) AND is_deleted = FALSE`,
        [ids]
      );

      if (degreeExists.length === 0) {
        return res
          .status(404)
          .json({ status: 404, error: { msg: "Degrees not found" } });
      }

      const [deleteResult] = await con.query(
        `UPDATE degrees SET is_deleted = TRUE WHERE degree_id IN (?)`,
        [ids]
      );

      if (deleteResult.affectedRows > 0) {
        return res
          .status(200)
          .json({ status: 200, msg: "Degrees deleted successfully" });
      } else {
        return res.status(404).json({
          status: 404,
          error: { msg: "No matching records found to delete" },
        });
      }
    } catch (error) {
      console.error("Error:", error);
      if (error.errno === 1451) {
        return res.status(409).json({
          error: {
            msg: "Cannot delete degree(s) because they are referenced in other records",
          },
        });
      }
      return res.status(500).json({ error: { msg: "Internal server error" } });
    } finally {
      con.release();
    }
  } catch (error) {
    console.error("Database connection error:", error);
    return res.status(500).json({ error: { msg: "Internal server error" } });
  }
};
