const pool = require("../config/db.js");

module.exports.createDegreeSubject = async (req, res) => {
  try {
    const {
      degree_id,
      degree_semester_id,
      subject_id,
      is_elective = false,
      total_marks = null,
    } = req.body;
    const con = await pool.getConnection();

    try {
      // Check if subject exists (either deleted or not)
      const [exists] = await con.query(
        `SELECT * FROM degree_subjects WHERE degree_semester_id = ? AND subject_id = ?`,
        [degree_semester_id, subject_id]
      );

      if (exists.length > 0) {
        const existingSubject = exists[0];

        if (existingSubject.is_deleted) {
          // If subject is soft deleted, reactivate it
          await con.query(
            `UPDATE degree_subjects
               SET is_deleted = FALSE, is_elective = ?, degree_id = ?, total_marks = ?
               WHERE degree_subject_id = ?`,
            [is_elective, degree_id, total_marks, existingSubject.degree_subject_id]
          );

          return res.status(200).json({
            status: 200,
            success: { msg: "Degree subject reactivated successfully." },
          });
        } else {
          // Subject already active
          return res.status(400).json({
            status: 400,
            error: { msg: "Degree subject already exists for this semester!" },
          });
        }
      }

      // If not found, create new record
      await con.query(
        `INSERT INTO degree_subjects (degree_id, degree_semester_id, subject_id, is_elective, total_marks)
           VALUES (?, ?, ?, ?, ?)`,
        [degree_id, degree_semester_id, subject_id, is_elective, total_marks]
      );

      return res.status(200).json({
        status: 200,
        success: { msg: "Degree subject created successfully." },
      });
    } catch (error) {
      console.error("Error:", error);
      return res.status(500).json({ error: { msg: "Internal server error" } });
    } finally {
      con.release();
    }
  } catch (error) {
    console.error("Database connection error:", error);
    return res.status(500).json({ error: { msg: "Internal server error" } });
  }
};

module.exports.updateDegreeSubject = async (req, res) => {
  try {
    const {
      degree_subject_id,
      degree_id,
      degree_semester_id,
      subject_id,
      is_elective = false,
      total_marks = null,
    } = req.body;
    const con = await pool.getConnection();

    try {
      // Check if new combination would violate UNIQUE constraint
      const [exists] = await con.query(
        `SELECT * FROM degree_subjects WHERE degree_semester_id = ? AND subject_id = ? AND degree_subject_id != ? AND is_deleted = FALSE`,
        [degree_semester_id, subject_id, degree_subject_id]
      );
      if (exists.length > 0) {
        return res.status(400).json({
          status: 400,
          error: {
            msg: "Another degree subject already exists for this semester!",
          },
        });
      }

      await con.query(
        `UPDATE degree_subjects SET degree_id = ?, degree_semester_id = ?, subject_id = ?, is_elective = ?, total_marks = ? WHERE degree_subject_id = ?`,
        [
          degree_id,
          degree_semester_id,
          subject_id,
          is_elective,
          total_marks,
          degree_subject_id,
        ]
      );

      return res.status(200).json({
        status: 200,
        success: { msg: "Degree subject updated successfully" },
      });
    } catch (error) {
      console.error("Error:", error);
      return res.status(500).json({ error: { msg: "Internal server error" } });
    } finally {
      con.release();
    }
  } catch (error) {
    console.error("Database connection error:", error);
    return res.status(500).json({ error: { msg: "Internal server error" } });
  }
};

module.exports.getAllDegreeSubjects = async (req, res) => {
  try {
    const { searchKeyWord = "", pageNumber = 1, pageSize = 10 } = req.body;
    const offset = (pageNumber - 1) * pageSize;
    const con = await pool.getConnection();

    try {
      const [totalResult] = await con.query(
        `SELECT COUNT(*) AS total 
           FROM degree_subjects ds
           JOIN subjects s ON s.subject_id = ds.subject_id
           WHERE ds.is_deleted = FALSE AND s.name LIKE ?`,
        [`%${searchKeyWord}%`]
      );

      const [degreeSubjects] = await con.query(
        `SELECT 
              ds.*, 
              s.name AS subject_name, 
              s.code AS subject_code,
              ds.total_marks AS total_marks
           FROM degree_subjects ds
           JOIN subjects s ON s.subject_id = ds.subject_id
           WHERE ds.is_deleted = FALSE AND s.name LIKE ?
           ORDER BY ds.degree_subject_id DESC
           LIMIT ? OFFSET ?`,
        [`%${searchKeyWord}%`, parseInt(pageSize), offset]
      );

      return res.status(200).json({
        status: 200,
        total: totalResult[0].total,
        pageSize,
        pageNumber,
        totalPages: Math.ceil(totalResult[0].total / pageSize),
        degreeSubjects,
      });
    } catch (error) {
      console.error("Error:", error);
      return res.status(500).json({ error: { msg: "Internal server error" } });
    } finally {
      con.release();
    }
  } catch (error) {
    console.error("Database connection error:", error);
    return res.status(500).json({ error: { msg: "Internal server error" } });
  }
};

module.exports.getDegreeSubjectById = async (req, res) => {
  try {
    const { id } = req.body;
    const con = await pool.getConnection();

    try {
      const [result] = await con.query(
        `SELECT * FROM degree_subjects WHERE degree_subject_id = ? AND is_deleted = 0`,
        [id]
      );
      return res.status(200).json({ status: 200, degreeSubject: result[0] });
    } catch (error) {
      console.error("Error:", error);
      return res.status(500).json({ error: { msg: "Internal server error" } });
    } finally {
      con.release();
    }
  } catch (error) {
    console.error("Database connection error:", error);
    return res.status(500).json({ error: { msg: "Internal server error" } });
  }
};

module.exports.deleteDegreeSubject = async (req, res) => {
  try {
    const { ids } = req.body; // expecting array of degree_subject_id
    const con = await pool.getConnection();

    try {
      const [exists] = await con.query(
        `SELECT * FROM degree_subjects WHERE degree_subject_id IN (?) AND is_deleted = FALSE`,
        [ids]
      );

      if (exists.length === 0) {
        return res
          .status(404)
          .json({ status: 404, error: { msg: "Degree subjects not found" } });
      }

      const [deleteResult] = await con.query(
        `UPDATE degree_subjects SET is_deleted = TRUE WHERE degree_subject_id IN (?)`,
        [ids]
      );

      if (deleteResult.affectedRows > 0) {
        return res
          .status(200)
          .json({ status: 200, msg: "Degree subjects deleted successfully" });
      } else {
        return res.status(404).json({
          status: 404,
          error: { msg: "No matching records found to delete" },
        });
      }
    } catch (error) {
      console.error("Error:", error);
      return res.status(500).json({ error: { msg: "Internal server error" } });
    } finally {
      con.release();
    }
  } catch (error) {
    console.error("Database connection error:", error);
    return res.status(500).json({ error: { msg: "Internal server error" } });
  }
};
