const pool = require("../config/db.js");

module.exports.createExam = async (req, res) => {
  let connection;
  try {
    const {
      name,
      total_marks,
      passing_marks,
      exam_date,
      edu_typ_id,
      class_id,
      session_id,
      batch_id,
      degree_id,
    } = req.body;


    if (!name || !total_marks || !passing_marks || !exam_date || !edu_typ_id) {
      return res.status(400).json({
        status: 400,
        error: {
          msg: "Missing required fields: name, total_marks, passing_marks, exam_date, edu_typ_id",
        },
      });
    }

    connection = await pool.getConnection();

    const [existingExam] = await connection.query(
      "SELECT exam_id FROM exams WHERE name = ?",
      [name]
    );

    if (existingExam.length > 0) {
      return res.status(409).json({
        status: 409,
        error: { msg: "An exam with this name already exists" },
      });
    }

    const [result] = await connection.query(
      `INSERT INTO exams (name, total_marks, passing_marks, exam_date, edu_typ_id, class_id, session_id, batch_id, degree_id) 
       VALUES (?, ?, ?, ?, ?, ?, ?, ?, ?)`,
      [
        name,
        total_marks,
        passing_marks,
        exam_date,
        edu_typ_id,
        class_id,
        session_id,
        batch_id,
        degree_id,
      ]
    );

    return res.status(201).json({
      status: 201,
      success: {
        msg: "Exam created successfully",
        exam_id: result.insertId,
      },
    });
  } catch (error) {
    console.error("Error creating exam:", error);
    return res.status(500).json({
      status: 500,
      error: { msg: "Internal server error" },
    });
  } finally {
    if (connection) connection.release();
  }
};

module.exports.getExamById = async (req, res) => {
  let connection;
  try {
    const { exam_id } = req.body;

    if (!exam_id) {
      return res.status(400).json({
        status: 400,
        error: { msg: "Exam ID is required" },
      });
    }

    connection = await pool.getConnection();

    const [exam] = await connection.query(
      "SELECT * FROM exams WHERE exam_id = ?",
      [exam_id]
    );

    if (exam.length === 0) {
      return res.status(404).json({
        status: 404,
        error: { msg: "Exam not found" },
      });
    }

    return res.status(200).json({
      status: 200,
      exam: exam[0],
    });
  } catch (error) {
    console.error("Error fetching exam:", error);
    return res.status(500).json({
      status: 500,
      error: { msg: "Internal server error" },
    });
  } finally {
    if (connection) connection.release();
  }
};

module.exports.getAllExams = async (req, res) => {
  let connection;

  try {
    const {
      searchKeyWord = "",
      edu_typ_id,
      class_id,
      session_id,
      batch_id,
      degree_id,
      pageNumber = 1,
      pageSize = 10,
    } = req.body;

    const offset = (pageNumber - 1) * pageSize;
    connection = await pool.getConnection();

    let whereClause = "WHERE 1=1";
    const params = [];

    if (searchKeyWord) {
      whereClause += " AND e.name LIKE ?";
      params.push(`%${searchKeyWord}%`);
    }
    if (edu_typ_id) {
      whereClause += " AND e.edu_typ_id = ?";
      params.push(edu_typ_id);
    }
    if (class_id) {
      whereClause += " AND e.class_id = ?";
      params.push(class_id);
    }
    if (session_id) {
      whereClause += " AND e.session_id = ?";
      params.push(session_id);
    }
    if (batch_id) {
      whereClause += " AND e.batch_id = ?";
      params.push(batch_id);
    }
    if (degree_id) {
      whereClause += " AND e.degree_id = ?";
      params.push(degree_id);
    }

    const [totalResult] = await connection.query(
      `SELECT COUNT(*) as total FROM exams e ${whereClause}`,
      params
    );

    params.push(parseInt(pageSize), offset);

    const [exams] = await connection.query(
      `SELECT e.*
       FROM exams e
       ${whereClause}
       ORDER BY e.created_at DESC 
       LIMIT ? OFFSET ?`,
      params
    );

    return res.status(200).json({
      status: 200,
      total: totalResult[0].total,
      pageSize,
      pageNumber,
      totalPages: Math.ceil(totalResult[0].total / pageSize),
      exams,
    });
  } catch (error) {
    console.error("Error fetching exams:", error);
    return res.status(500).json({
      status: 500,
      error: { msg: "Internal server error" },
    });
  } finally {
    if (connection) connection.release();
  }
};

module.exports.updateExam = async (req, res) => {
  let connection;
  try {
    const { exam_id } = req.body;

    if (!exam_id) {
      return res.status(400).json({
        status: 400,
        error: { msg: "Exam ID is required" },
      });
    }

    connection = await pool.getConnection();

    const [existingExam] = await connection.query(
      "SELECT * FROM exams WHERE exam_id = ?",
      [exam_id]
    );

    if (existingExam.length === 0) {
      return res.status(404).json({
        status: 404,
        error: { msg: "Exam not found" },
      });
    }

    if (req.body.name && req.body.name !== existingExam[0].name) {
      const [nameExists] = await connection.query(
        "SELECT exam_id FROM exams WHERE name = ? AND exam_id != ?",
        [req.body.name, exam_id]
      );

      if (nameExists.length > 0) {
        return res.status(409).json({
          status: 409,
          error: { msg: "Another exam with this name already exists" },
        });
      }
    }

    const updateFields = [];
    const updateParams = [];
    const allowedFields = [
      "name",
      "total_marks",
      "passing_marks",
      "exam_date",
      "edu_typ_id",
      "class_id",
      "session_id",
      "batch_id",
      "degree_id",
    ];

    allowedFields.forEach((field) => {
      if (req.body.hasOwnProperty(field)) {
        updateFields.push(`${field} = ?`);
        updateParams.push(req.body[field]);
      }
    });

    if (updateFields.length === 0) {
      return res.status(400).json({
        status: 400,
        error: { msg: "No fields provided for update" },
      });
    }

    updateParams.push(exam_id);

    await connection.query(
      `UPDATE exams SET ${updateFields.join(", ")} WHERE exam_id = ?`,
      updateParams
    );

    return res.status(200).json({
      status: 200,
      success: { msg: "Exam updated successfully" },
    });
  } catch (error) {
    console.error("Error updating exam:", error);
    return res.status(500).json({
      status: 500,
      error: { msg: "Internal server error" },
    });
  } finally {
    if (connection) connection.release();
  }
};

module.exports.deleteExam = async (req, res) => {
  try {
    const { ids } = req.body;

    if (!Array.isArray(ids) || ids.length === 0) {
      return res.status(400).json({
        status: 400,
        error: { msg: "An array of exam IDs is required" },
      });
    }

    const con = await pool.getConnection();
    try {
      const [deleteResult] = await con.query(
        `DELETE FROM exams WHERE exam_id IN (?)`,
        [ids]
      );

      if (deleteResult.affectedRows > 0) {
        return res.status(200).json({
          status: 200,
          msg: "Exams deleted successfully",
        });
      } else {
        return res.status(404).json({
          status: 404,
          error: { msg: "No matching exams found to delete" },
        });
      }
    } catch (error) {
      console.error("Error:", error);
      if (error.code === "ER_ROW_IS_REFERENCED_2") {
        return res.status(400).json({
          status: 400,
          error: {
            msg: "Cannot delete. One or more exams are associated with other records.",
          },
        });
      }
      return res.status(500).json({ error: { msg: "Internal server error" } });
    } finally {
      con.release();
    }
  } catch (error) {
    console.error("Database connection error:", error);
    return res.status(500).json({ error: { msg: "Internal server error" } });
  }
};