const pool = require("../config/db.js");

module.exports.createExpense = async (req, res) => {
  let connection;
  try {
    const {
      description,
      amount,
      category_id,
      expense_date,
      payment_method,
      expense_for_type,
      expense_for_id,
      receipt_url,
      status = 'Paid',
      created_by, // Should ideally come from authenticated user session
    } = req.body;

    if (!description || !amount || !category_id || !expense_date || !payment_method) {
      return res.status(400).json({
        status: 400,
        error: {
          msg: "Missing required fields: description, amount, category_id, expense_date, payment_method",
        },
      });
    }

    connection = await pool.getConnection();

    const [result] = await connection.query(
      `INSERT INTO expenses (description, amount, category_id, expense_date, payment_method, expense_for_type, expense_for_id, receipt_url, status, created_by) 
       VALUES (?, ?, ?, ?, ?, ?, ?, ?, ?, ?)`,
      [
        description,
        amount,
        category_id,
        expense_date,
        payment_method,
        expense_for_type,
        expense_for_id,
        receipt_url,
        status,
        created_by
      ]
    );

    return res.status(201).json({
      status: 201,
      success: {
        msg: "Expense created successfully",
        expense_id: result.insertId,
      },
    });
  } catch (error) {
    console.error("Error creating expense:", error);
    return res.status(500).json({
      status: 500,
      error: { msg: "Internal server error" },
    });
  } finally {
    if (connection) connection.release();
  }
};

module.exports.getExpenseById = async (req, res) => {
  let connection;
  try {
    const { expense_id } = req.body;

    if (!expense_id) {
      return res.status(400).json({
        status: 400,
        error: { msg: "Expense ID is required" },
      });
    }

    connection = await pool.getConnection();

    const [expense] = await connection.query(
      `SELECT e.*, ec.name as category_name 
       FROM expenses e
       LEFT JOIN expense_categories ec ON e.category_id = ec.category_id
       WHERE e.expense_id = ?`,
      [expense_id]
    );

    if (expense.length === 0) {
      return res.status(404).json({
        status: 404,
        error: { msg: "Expense not found" },
      });
    }

    return res.status(200).json({
      status: 200,
      expense: expense[0],
    });
  } catch (error) {
    console.error("Error fetching expense:", error);
    return res.status(500).json({
      status: 500,
      error: { msg: "Internal server error" },
    });
  } finally {
    if (connection) connection.release();
  }
};

module.exports.getAllExpenses = async (req, res) => {
  let connection;
  try {
    const {
      searchKeyWord = "",
      category_id,
      expense_for_type,
      start_date,
      end_date,
      pageNumber = 1,
      pageSize = 10,
    } = req.body;

    const offset = (pageNumber - 1) * pageSize;
    connection = await pool.getConnection();

    let whereClause = "WHERE 1=1";
    const params = [];

    if (searchKeyWord) {
      whereClause += " AND e.description LIKE ?";
      params.push(`%${searchKeyWord}%`);
    }
    if (category_id) {
      whereClause += " AND e.category_id = ?";
      params.push(category_id);
    }
    if (expense_for_type) {
      whereClause += " AND e.expense_for_type = ?";
      params.push(expense_for_type);
    }
    if (start_date) {
      whereClause += " AND e.expense_date >= ?";
      params.push(start_date);
    }
    if (end_date) {
      whereClause += " AND e.expense_date <= ?";
      params.push(end_date);
    }

    const [totalResult] = await connection.query(
      `SELECT COUNT(*) as total FROM expenses e ${whereClause}`,
      params
    );

    params.push(parseInt(pageSize), offset);

    const [expenses] = await connection.query(
      `SELECT e.*, ec.name as category_name
       FROM expenses e
       LEFT JOIN expense_categories ec ON e.category_id = ec.category_id
       ${whereClause}
       ORDER BY e.expense_date DESC, e.created_at DESC
       LIMIT ? OFFSET ?`,
      params
    );

    return res.status(200).json({
      status: 200,
      total: totalResult[0].total,
      pageSize,
      pageNumber,
      totalPages: Math.ceil(totalResult[0].total / pageSize),
      expenses,
    });
  } catch (error) {
    console.error("Error fetching expenses:", error);
    return res.status(500).json({
      status: 500,
      error: { msg: "Internal server error" },
    });
  }
};

module.exports.updateExpense = async (req, res) => {
  let connection;
  try {
    const { expense_id } = req.body;

    if (!expense_id) {
      return res.status(400).json({
        status: 400,
        error: { msg: "Expense ID is required" },
      });
    }

    connection = await pool.getConnection();

    const [existingExpense] = await connection.query(
      "SELECT expense_id FROM expenses WHERE expense_id = ?",
      [expense_id]
    );

    if (existingExpense.length === 0) {
      return res.status(404).json({
        status: 404,
        error: { msg: "Expense not found" },
      });
    }

    const updateFields = [];
    const updateParams = [];
    const allowedFields = [
      "description", "amount", "category_id", "expense_date", "payment_method",
      "expense_for_type", "expense_for_id", "receipt_url", "status"
    ];

    allowedFields.forEach((field) => {
      if (req.body.hasOwnProperty(field)) {
        updateFields.push(`${field} = ?`);
        updateParams.push(req.body[field]);
      }
    });

    if (updateFields.length === 0) {
      return res.status(400).json({
        status: 400,
        error: { msg: "No fields provided for update" },
      });
    }

    updateParams.push(expense_id);

    await connection.query(
      `UPDATE expenses SET ${updateFields.join(", ")} WHERE expense_id = ?`,
      updateParams
    );

    return res.status(200).json({
      status: 200,
      success: { msg: "Expense updated successfully" },
    });
  } catch (error) {
    console.error("Error updating expense:", error);
    return res.status(500).json({
      status: 500,
      error: { msg: "Internal server error" },
    });
  } finally {
    if (connection) connection.release();
  }
};

module.exports.deleteExpense = async (req, res) => {
  try {
    const { ids } = req.body;

    if (!Array.isArray(ids) || ids.length === 0) {
      return res.status(400).json({
        status: 400,
        error: { msg: "An array of expense IDs is required" },
      });
    }

    const con = await pool.getConnection();
    try {
      const [deleteResult] = await con.query(
        `DELETE FROM expenses WHERE expense_id IN (?)`,
        [ids]
      );

      if (deleteResult.affectedRows > 0) {
        return res.status(200).json({
          status: 200,
          msg: "Expenses deleted successfully",
        });
      } else {
        return res.status(404).json({
          status: 404,
          error: { msg: "No matching expenses found to delete" },
        });
      }
    } catch (error) {
      console.error("Error deleting expenses:", error);
      return res.status(500).json({ error: { msg: "Internal server error" } });
    } finally {
      con.release();
    }
  } catch (error) {
    console.error("Database connection error:", error);
    return res.status(500).json({ error: { msg: "Internal server error" } });
  }
};
module.exports.getAllExpenseCategories = async (req, res) => {
    let connection;
    try {
        connection = await pool.getConnection();
        
            // This query fetches all categories, ordered by name, which is ideal for a dropdown.
        const [categories] = await connection.query(
            "SELECT category_id, name FROM expense_categories ORDER BY name ASC"
        );

        return res.status(200).json({
            status: 200,
            categories: categories
        });

    } catch (error) {
        console.error("Error fetching expense categories:", error);
        return res.status(500).json({ status: 500, error: { msg: "Internal server error" } });
    } finally {
        if (connection) connection.release();
    }
};