const express = require("express");
const router = express.Router();
const pool = require("../config/db");

// Create student receiver
module.exports.createReceiver = async (req, res) => {
  try {
    const { student_id, relation_id, name, cnic, mobile_number } = req.body;
    const sql = `INSERT INTO student_receivers (student_id, relation_id, name, cnic, mobile_number, created_at) VALUES (?, ?, ?, ?, ?, NOW())`;
    const [result] = await pool.execute(sql, [
      student_id,
      relation_id,
      name,
      cnic,
      mobile_number,
    ]);
    res.status(201).json({
      success: true,
      id: result.insertId,
      message: "Student`s receiver successfully stored!",
    });
  } catch (error) {
    res.status(500).json({ success: false, error: error.message });
  }
};

// Get all student receivers
module.exports.getAllReceivers = async (req, res) => {
  try {
    const {
      searchKeyWord = "",
      student_id,
      pageNumber = 1,
      pageSize = 10,
    } = req.body;
    const offset = (pageNumber - 1) * pageSize;
    const con = await pool.getConnection();
    try {
      const [totalResult] = await con.query(
        `SELECT COUNT(*) AS total FROM student_receivers WHERE name LIKE ? AND student_id = ? AND deleted_at IS NULL`,
        [`%${searchKeyWord}%`, student_id]
      );

      const [receivers] = await con.query(
        `SELECT * FROM student_receivers WHERE name LIKE ? AND student_id = ? AND deleted_at IS NULL ORDER BY id DESC LIMIT ? OFFSET ?`,
        [`%${searchKeyWord}%`, student_id, parseInt(pageSize), offset]
      );

      return res.status(200).json({
        status: 200,
        total: totalResult[0].total,
        pageSize,
        pageNumber,
        totalPages: Math.ceil(totalResult[0].total / pageSize),
        receivers,
      });
    } catch (error) {
      console.error("Error:", error);
      return res.status(500).json({ error: { msg: "Internal server error" } });
    } finally {
      con.release();
    }
  } catch (error) {
    console.error("Database connection error:", error);
    return res.status(500).json({ error: { msg: "Internal server error" } });
  }
};

// Get student receiver by ID
module.exports.getReceiverById = async (req, res) => {
  try {
    const { id, student_id } = req.body;
    const sql =
      "SELECT * FROM student_receivers WHERE id = ? AND student_id = ? AND deleted_at IS NULL";
    const [receiver] = await pool.execute(sql, [id, student_id]);
    // if (receiver.length === 0) {
    //     return res.status(404).json({ success: false, message: "Receiver not found" });
    // }
    res.json({ success: true, data: { receiver: receiver[0] } });
  } catch (error) {
    res.status(500).json({ success: false, error: error.message });
  }
};

// Update student receiver
module.exports.updateReceiver = async (req, res) => {
  try {
    const { name, cnic, mobile_number, id, student_id, relation_id } = req.body;
    const sql = `
            UPDATE student_receivers 
            SET name = ?, cnic = ?, mobile_number = ?, updated_at = NOW(), relation_id = ? 
            WHERE id = ? AND student_id = ? AND deleted_at IS NULL
        `;
    const [result] = await pool.execute(sql, [
      name,
      cnic,
      mobile_number,
      relation_id,
      id,
      student_id,
    ]);

    if (result.affectedRows === 0) {
      return res
        .status(404)
        .json({ success: false, message: "Receiver not found" });
    }
    res.json({
      success: true,
      message: "Student`s receiver updated successfully!",
    });
  } catch (error) {
    res.status(500).json({ success: false, error: error.message });
  }
};

// Soft delete student receiver
module.exports.deleteReceiver = async (req, res) => {
  try {
    const { ids } = req.body;
    const con = await pool.getConnection();
    const [result] = await con.query(
      `UPDATE student_receivers SET deleted_at = NOW() WHERE id IN (?) AND deleted_at IS NULL`,
      [ids]
    );

    if (result.affectedRows === 0) {
      return res.status(404).json({
        success: false,
        message: "Receiver not found or already deleted",
      });
    }
    res
      .status(200)
      .json({ success: true, msg: "Receiver deleted successfully" });
  } catch (error) {
    res.status(500).json({ success: false, error: error.message });
  }
};
