const pool = require("../config/db.js");

module.exports.createSection = async (req, res) => {
  try {
    const { name } = req.body;
    const con = await pool.getConnection();
    try {
      // Check if section with this name already exists
      const [sectionExists] = await con.query(
        `SELECT * FROM section WHERE section_name = ? AND deleted_at IS NULL`,
        [name]
      );
      if (sectionExists.length > 0) {
        return res.status(400).json({
          status: 400,
          error: { msg: "Section with this name already exists!" },
        });
      }
      await con.query(
        `INSERT INTO section (section_name, created_at, updated_at) VALUES (?, NOW(), NOW())`,
        [name]
      );
      return res.status(200).json({
        status: 200,
        success: { msg: "Section created successfully" },
      });
    } catch (error) {
      console.error("Error:", error);
      return res.status(500).json({ error: { msg: "Internal server error" } });
    } finally {
      con.release();
    }
  } catch (error) {
    console.error("Database connection error:", error);
    return res.status(500).json({ error: { msg: "Internal server error" } });
  }
};

module.exports.updateSection = async (req, res) => {
  try {
    const { section_id, name } = req.body;
    const con = await pool.getConnection();
    try {
      // Check if another section with this name exists
      const [sectionExists] = await con.query(
        `SELECT * FROM section WHERE section_name = ? AND section_id != ? AND deleted_at IS NULL`,
        [name, section_id]
      );
      if (sectionExists.length > 0) {
        return res.status(400).json({
          status: 400,
          error: { msg: "Another section with this name already exists!" },
        });
      }
      await con.query(
        `UPDATE section SET section_name = ?, updated_at = NOW() WHERE section_id = ?`,
        [name, section_id]
      );
      return res.status(200).json({
        status: 200,
        success: { msg: "Section updated successfully" },
      });
    } catch (error) {
      console.error("Error:", error);
      return res.status(500).json({ error: { msg: "Internal server error" } });
    } finally {
      con.release();
    }
  } catch (error) {
    console.error("Database connection error:", error);
    return res.status(500).json({ error: { msg: "Internal server error" } });
  }
};

module.exports.getAllSections = async (req, res) => {
  try {
    const { searchKeyWord = "", pageNumber = 1, pageSize = 10 } = req.body;
    const offset = (pageNumber - 1) * pageSize;
    const con = await pool.getConnection();
    try {
      const [totalResult] = await con.query(
        `SELECT COUNT(*) AS total FROM section WHERE section_name LIKE ? AND deleted_at IS NULL`,
        [`%${searchKeyWord}%`]
      );
      const [sections] = await con.query(
        `SELECT section_id, section_name, created_at, updated_at, deleted_at FROM section WHERE section_name LIKE ? AND deleted_at IS NULL ORDER BY section_id DESC LIMIT ? OFFSET ?`,
        [`%${searchKeyWord}%`, parseInt(pageSize), offset]
      );
      return res.status(200).json({
        status: 200,
        total: totalResult[0].total,
        pageSize,
        pageNumber,
        totalPages: Math.ceil(totalResult[0].total / pageSize),
        sections,
      });
    } catch (error) {
      console.error("Error:", error);
      return res.status(500).json({ error: { msg: "Internal server error" } });
    } finally {
      con.release();
    }
  } catch (error) {
    console.error("Database connection error:", error);
    return res.status(500).json({ error: { msg: "Internal server error" } });
  }
};

module.exports.getSectionById = async (req, res) => {
  try {
    const { section_id } = req.body;
    const con = await pool.getConnection();
    try {
      const [result] = await con.query(
        `SELECT section_id, section_name, created_at, updated_at, deleted_at FROM section WHERE section_id = ? AND deleted_at IS NULL`,
        [section_id]
      );
      return res.status(200).json({ status: 200, section: result });
    } catch (error) {
      console.error("Error:", error);
      return res.status(500).json({ error: { msg: "Internal server error" } });
    } finally {
      con.release();
    }
  } catch (error) {
    console.error("Database connection error:", error);
    return res.status(500).json({ error: { msg: "Internal server error" } });
  }
};

module.exports.deleteSection = async (req, res) => {
  try {
    const { ids } = req.body; // expecting array of section_ids
    const con = await pool.getConnection();
    try {
      const [sectionExists] = await con.query(
        `SELECT * FROM section WHERE section_id IN (?) AND deleted_at IS NULL`,
        [ids]
      );
      if (sectionExists.length === 0) {
        return res.status(404).json({ status: 404, error: { msg: "Sections not found" } });
      }
      const [deleteResult] = await con.query(
        `UPDATE section SET deleted_at = NOW() WHERE section_id IN (?)`,
        [ids]
      );
      if (deleteResult.affectedRows > 0) {
        return res.status(200).json({ status: 200, msg: "Sections deleted successfully" });
      } else {
        return res.status(404).json({ status: 404, error: { msg: "No matching records found to delete" } });
      }
    } catch (error) {
      console.error("Error:", error);
      return res.status(500).json({ error: { msg: "Internal server error" } });
    } finally {
      con.release();
    }
  } catch (error) {
    console.error("Database connection error:", error);
    return res.status(500).json({ error: { msg: "Internal server error" } });
  }
};

module.exports.getSectionDropdown = async (req, res) => {
  try {
    const con = await pool.getConnection();
    try {
      const [sections] = await con.query(
        `SELECT section_id, section_name FROM section WHERE deleted_at IS NULL ORDER BY section_name`
      );
      return res.status(200).json({ status: 200, sections });
    } catch (error) {
      console.error("Error:", error);
      return res.status(500).json({ error: { msg: "Internal server error" } });
    } finally {
      con.release();
    }
  } catch (error) {
    console.error("Database connection error:", error);
    return res.status(500).json({ error: { msg: "Internal server error" } });
  }
};  