const pool = require("../config/db.js");
const { SemesterDuration } = require("../constants/enum.js");

// Create Semester
module.exports.createSemester = async (req, res) => {
  try {
    const { name, degree_id, sequence_number, duration_in_weeks } = req.body;
    const con = await pool.getConnection();

    try {
      const [exists] = await con.query(
        `SELECT * FROM degree_semesters WHERE degree_id = ? AND sequence_number = ?`,
        [degree_id, sequence_number]
      );

      if (exists.length > 0) {
        return res.status(400).json({
          status: 400,
          error: {
            msg: "Semester with this sequence number already exists for this degree!",
          },
        });
      }

      await con.query(
        `INSERT INTO degree_semesters (degree_id, name, sequence_number, duration_in_weeks) VALUES (?, ?, ?, ?)`,
        [degree_id, name, sequence_number, duration_in_weeks]
      );

      return res.status(200).json({
        status: 200,
        success: { msg: "Semester created successfully" },
      });
    } catch (error) {
      console.error("Error:", error);
      return res.status(500).json({ error: { msg: "Internal server error" } });
    } finally {
      con.release();
    }
  } catch (error) {
    console.error("Database connection error:", error);
    return res.status(500).json({ error: { msg: "Internal server error" } });
  }
};

// Update Semester
module.exports.updateSemester = async (req, res) => {
  try {
    const {
      id,
      name,
      degree_id,
      sequence_number,
      duration_in_weeks,
    } = req.body;
    const con = await pool.getConnection();

    try {
      const [exists] = await con.query(
        `SELECT * FROM degree_semesters WHERE degree_id = ? AND sequence_number = ? AND degree_semester_id != ?`,
        [degree_id, sequence_number, id]
      );
      
      if (exists.length > 0) {
        return res.status(400).json({
          status: 400,
          error: {
            msg: "Another semester with this sequence number already exists for this degree!",
          },
        });
      }

      await con.query(
        `UPDATE degree_semesters SET name = ?, sequence_number = ?, duration_in_weeks = ? WHERE degree_semester_id = ? AND degree_id = ?`,
        [
          name,
          sequence_number,
          duration_in_weeks,
          id,
          degree_id,
        ]
      );

      return res.status(200).json({
        status: 200,
        success: { msg: "Semester updated successfully" },
      });
    } catch (error) {
      console.error("Error:", error);
      return res.status(500).json({ error: { msg: "Internal server error" } });
    } finally {
      con.release();
    }
  } catch (error) {
    console.error("Database connection error:", error);
    return res.status(500).json({ error: { msg: "Internal server error" } });
  }
};

// Get All Semesters
module.exports.getAllSemesters = async (req, res) => {
  try {
    const {
      searchKeyWord = "",
      degree_id,
      pageNumber = 1,
      pageSize = 10,
    } = req.body;
    const offset = (pageNumber - 1) * pageSize;
    const con = await pool.getConnection();

    try {
      const [totalResult] = await con.query(
        `SELECT COUNT(*) AS total FROM degree_semesters WHERE name LIKE ? AND degree_id = ? AND is_deleted = FALSE`,
        [`%${searchKeyWord}%`, degree_id]
      );

      const [semesters] = await con.query(
        `SELECT * FROM degree_semesters WHERE name LIKE ? AND degree_id = ? AND is_deleted = FALSE ORDER BY sequence_number ASC LIMIT ? OFFSET ?`,
        [`%${searchKeyWord}%`, degree_id, parseInt(pageSize), offset]
      );

      return res.status(200).json({
        status: 200,
        total: totalResult[0].total,
        pageSize,
        pageNumber,
        totalPages: Math.ceil(totalResult[0].total / pageSize),
        semesters,
      });
    } catch (error) {
      console.error("Error:", error);
      return res.status(500).json({ error: { msg: "Internal server error" } });
    } finally {
      con.release();
    }
  } catch (error) {
    console.error("Database connection error:", error);
    return res.status(500).json({ error: { msg: "Internal server error" } });
  }
};

// Get Semester By ID
module.exports.getSemesterById = async (req, res) => {
  try {
    const { degree_semester_id } = req.body;
    const con = await pool.getConnection();

    try {
      const [result] = await con.query(
        `SELECT * FROM degree_semesters WHERE degree_semester_id = ? AND is_deleted = FALSE`,
        [degree_semester_id]
      );

      return res.status(200).json({ status: 200, semester: result });
    } catch (error) {
      console.error("Error:", error);
      return res.status(500).json({ error: { msg: "Internal server error" } });
    } finally {
      con.release();
    }
  } catch (error) {
    console.error("Database connection error:", error);
    return res.status(500).json({ error: { msg: "Internal server error" } });
  }
};

// Delete Semesters (Soft Delete with usage check)
module.exports.deleteSemesters = async (req, res) => {
  try {
    const { ids } = req.body;
    const con = await pool.getConnection();

    try {
      // 1. Check if semesters exist
      const [exists] = await con.query(
        `SELECT degree_semester_id FROM degree_semesters WHERE degree_semester_id IN (?) AND is_deleted = FALSE`,
        [ids]
      );

      if (exists.length === 0) {
        return res.status(404).json({
          status: 404,
          error: { msg: "Semesters not found." },
        });
      }

      // 2. Check if semesters are being used in degree_subjects (or any other related tables)
      const [inUse] = await con.query(
        `SELECT DISTINCT semester_id FROM degree_subjects WHERE semester_id IN (?) AND is_deleted = FALSE`,
        [ids]
      );

      if (inUse.length > 0) {
        return res.status(400).json({
          status: 400,
          error: { msg: "Cannot delete. One or more semesters are in use." },
          usedSemesterIds: inUse.map((row) => row.semester_id),
        });
      }

      // 3. Safe to delete (soft delete)
      await con.query(
        `UPDATE degree_semesters SET is_deleted = TRUE WHERE degree_semester_id IN (?)`,
        [ids]
      );

      return res.status(200).json({
        status: 200,
        success: { msg: "Semesters deleted successfully." },
      });
    } catch (error) {
      console.error("Error:", error);
      return res.status(500).json({ error: { msg: "Internal server error" } });
    } finally {
      con.release();
    }
  } catch (error) {
    console.error("Database connection error:", error);
    return res.status(500).json({ error: { msg: "Internal server error" } });
  }
};

// Get Semester Durations
module.exports.getDurations = async (req, res) => {
  try {
    const durations = Object.values(SemesterDuration);
    res.status(200).json(durations);
  } catch (error) {
    console.error("Error fetching durations:", error);
    res.status(500).json({ msg: "Failed to fetch semester durations" });
  }
};
