const pool = require("../config/db.js");

// Create a new session
module.exports.createSession = async (req, res) => {
  let connection;
  try {
    const {
      session_name,
      start_date,
      end_date,
      is_active = 0
    } = req.body;

    // Validation
    if (!session_name || !start_date || !end_date) {
      return res.status(400).json({
        status: 400,
        error: { msg: "Missing required fields: session_name, start_date, end_date" }
      });
    }

    connection = await pool.getConnection();

    // Check if session name already exists
    const [existingSession] = await connection.query(
      "SELECT * FROM sessions WHERE session_name = ?",
      [session_name]
    );

    if (existingSession.length > 0) {
      return res.status(409).json({
        status: 409,
        error: { msg: "A session with this name already exists" }
      });
    }

    // If is_active is set, deactivate all other sessions
    if (is_active == 1 || is_active === true) {
      await connection.query(
        "UPDATE sessions SET is_active = 0 WHERE is_active = 1"
      );
    }

    // Insert new session
    const [result] = await connection.query(
      `INSERT INTO sessions (session_name, start_date, end_date, is_active) 
       VALUES (?, ?, ?, ?)`,
      [session_name, start_date, end_date, is_active]
    );

    return res.status(201).json({
      status: 201,
      success: { 
        msg: "Session created successfully",
        session_id: result.insertId
      }
    });
  } catch (error) {
    console.error("Error creating session:", error);
    return res.status(500).json({
      status: 500,
      error: { msg: "Internal server error" }
    });
  } finally {
    if (connection) connection.release();
  }
};

// Get a session by ID
module.exports.getSessionById = async (req, res) => {
  let connection;
  try {
    const { session_id } = req.body;
    

    if (!session_id) {
      return res.status(400).json({
        status: 400,
        error: { msg: "Session ID is required" }
      });
    }

    connection = await pool.getConnection();
    
    const [session] = await connection.query(
      "SELECT * FROM sessions WHERE session_id = ? AND deleted_at is null",
      [session_id]
    );

    if (session.length === 0) {
      return res.status(404).json({
        status: 404,
        error: { msg: "Session not found" }
      });
    }

    return res.status(200).json({
      status: 200,
      session: session[0]
    });
  } catch (error) {
    console.error("Error fetching session:", error);
    return res.status(500).json({
      status: 500,
      error: { msg: "Internal server error" }
    });
  } finally {
    if (connection) connection.release();
  }
};

// Get all sessions with pagination and filtering
module.exports.getAllSessions = async (req, res) => {
  let connection;
 
  try {
    const {
      searchKeyWord = "",
      is_active,
      pageNumber = 1,
      pageSize = 10
    } = req.body;
    
    const offset = (pageNumber - 1) * pageSize;
    connection = await pool.getConnection();

    // Build the WHERE clause based on filters
    let whereClause = "WHERE 1=1";
    const params = [];

    if (searchKeyWord) {
      whereClause += " AND session_name LIKE ?";
      params.push(`%${searchKeyWord}%`);
    }

    if (typeof is_active !== 'undefined') {
      whereClause += " AND is_active = ?";
      params.push(is_active);
    }

    // Count total sessions matching the filters
    const [totalResult] = await connection.query(
      `SELECT COUNT(*) as total FROM sessions ${whereClause} AND deleted_at is null`,
      params
    );

    // Get paginated sessions
    params.push(parseInt(pageSize), offset);
    const [sessions] = await connection.query(
      `SELECT * FROM sessions 
       ${whereClause} AND deleted_at is null
       ORDER BY created_at DESC 
       LIMIT ? OFFSET ?`,
      params 
    );
    return res.status(200).json({
      status: 200,
      total: totalResult[0].total,
      pageSize,
      pageNumber,
      totalPages: Math.ceil(totalResult[0].total / pageSize),
      sessions
    });
  } catch (error) {
    console.error("Error fetching sessions:", error);
    return res.status(500).json({
      status: 500,
      error: { msg: "Internal server error" }
    });
  } finally {
    if (connection) connection.release();
  }
};

// Update a session
module.exports.updateSession = async (req, res) => {
  let connection;
  try {
    const {
      session_id,
      session_name,
      start_date,
      end_date,
      is_active
    } = req.body;

    if (!session_id) {
      return res.status(400).json({
        status: 400,
        error: { msg: "Session ID is required" }
      });
    }

    connection = await pool.getConnection();

    // Check if session exists
    const [existingSession] = await connection.query(
      "SELECT * FROM sessions WHERE session_id = ? AND deleted_at is null",
      [session_id]
    );

    if (existingSession.length === 0) {
      return res.status(404).json({
        status: 404,
        error: { msg: "Session not found" }
      });
    }

    // Check if updated name already exists for another session
    if (session_name && session_name !== existingSession[0].session_name) {
      const [nameExists] = await connection.query(
        "SELECT * FROM sessions WHERE session_name = ? AND session_id != ? AND deleted_at is null",
        [session_name, session_id]
      );

      if (nameExists.length > 0) {
        return res.status(409).json({
          status: 409,
          error: { msg: "Another session with this name already exists" }
        });
      }
    }

    // If is_active is set to 1, deactivate all other sessions
    if (typeof is_active !== 'undefined' && (is_active == 1 || is_active === true)) {
      await connection.query(
        "UPDATE sessions SET is_active = 0 WHERE is_active = 1 AND session_id != ?",
        [session_id]
      );
    }

    // Build UPDATE query dynamically based on provided fields
    const updateFields = [];
    const updateParams = [];

    if (session_name) {
      updateFields.push("session_name = ?");
      updateParams.push(session_name);
    }

    if (start_date) {
      updateFields.push("start_date = ?");
      updateParams.push(start_date);
    }

    if (end_date) {
      updateFields.push("end_date = ?");
      updateParams.push(end_date);
    }

    if (typeof is_active !== 'undefined') {
      updateFields.push("is_active = ?");
      updateParams.push(is_active);
    }

    if (updateFields.length === 0) {
      return res.status(400).json({
        status: 400,
        error: { msg: "No fields provided for update" }
      });
    }

    // Add session_id to params for WHERE clause
    updateParams.push(session_id);

    // Execute update
    const [updateResult] = await connection.query(
      `UPDATE sessions SET ${updateFields.join(", ")} WHERE session_id = ?`,
      updateParams
    );

    if (updateResult.affectedRows === 0) {
      return res.status(500).json({
        status: 500,
        error: { msg: "Failed to update session" }
      });
    }

    return res.status(200).json({
      status: 200,
      success: { msg: "Session updated successfully" }
    });
  } catch (error) {
    console.error("Error updating session:", error);
    return res.status(500).json({
      status: 500,
      error: { msg: "Internal server error" }
    });
  } finally {
    if (connection) connection.release();
  }
};

// Delete a session
module.exports.deleteSession = async (req, res) => {
  try {
    const { ids } = req.body; // expecting array of session_ids

    if (!Array.isArray(ids) || ids.length === 0) {
      return res.status(400).json({
        status: 400,
        error: { msg: "An array of session IDs is required" }
      });
    }

    const con = await pool.getConnection();
    try {
      // Check if sessions exist and are not already deleted
      const [sessionsExist] = await con.query(
        `SELECT * FROM sessions WHERE session_id IN (?) AND deleted_at IS NULL`,
        [ids]
      );

      if (sessionsExist.length === 0) {
        return res.status(404).json({
          status: 404,
          error: { msg: "Sessions not found" }
        });
      }

      // Soft delete sessions
      const [deleteResult] = await con.query(
        `UPDATE sessions SET deleted_at = NOW() WHERE session_id IN (?)`,
        [ids]
      );

      if (deleteResult.affectedRows > 0) {
        return res.status(200).json({
          status: 200,
          msg: "Sessions deleted successfully"
        });
      } else {
        return res.status(404).json({
          status: 404,
          error: { msg: "No matching sessions found to delete" }
        });
      }
    } catch (error) {
      console.error("Error:", error);
      return res.status(500).json({ error: { msg: "Internal server error" } });
    } finally {
      con.release();
    }
  } catch (error) {
    console.error("Database connection error:", error);
    return res.status(500).json({ error: { msg: "Internal server error" } });
  }
};

module.exports.getCurrentActiveSession = async (req, res) => {
  let connection;
  try {
    connection = await pool.getConnection();

    const [activeSessions] = await connection.query(
      "SELECT * FROM sessions WHERE is_active = 1 AND deleted_at IS NULL ORDER BY created_at DESC LIMIT 1"
    );

    if (activeSessions.length === 0) {
      return res.status(404).json({
        status: 404,
        error: { msg: "No active session found" }
      });
    }

    return res.status(200).json({
      status: 200,
      session: activeSessions[0]
    });
  } catch (error) {
    console.error("Error fetching current active session:", error);
    return res.status(500).json({
      status: 500,
      error: { msg: "Internal server error" }
    });
  } finally {
    if (connection) connection.release();
  }
};
