const pool = require("../config/db.js");

module.exports.createSubject = async (req, res) => {
  try {
    const {
      name,
      code,
      description = null,
      learning_outcomes = null,
      prerequisite_subject_id = null,
    } = req.body;
    const con = await pool.getConnection();

    try {
      const [subjectExists] = await con.query(
        `SELECT * FROM subjects WHERE code = ?`,
        [code]
      );
      if (subjectExists.length > 0) {
        return res.status(400).json({
          status: 400,
          error: { msg: "Subject with this code already exists!" },
        });
      }

      await con.query(
        `INSERT INTO subjects (name, code, description, learning_outcomes, prerequisite_subject_id) VALUES (?, ?, ?, ?, ?)`,
        [name, code, description, learning_outcomes, prerequisite_subject_id]
      );

      return res.status(200).json({
        status: 200,
        success: { msg: "Subject created successfully" },
      });
    } catch (error) {
      console.error("Error:", error);
      return res.status(500).json({ error: { msg: "Internal server error" } });
    } finally {
      con.release();
    }
  } catch (error) {
    console.error("Database connection error:", error);
    return res.status(500).json({ error: { msg: "Internal server error" } });
  }
};
module.exports.updateSubject = async (req, res) => {
  try {
    const {
      subject_id,
      name,
      code,
      description = null,
      learning_outcomes = null,
      prerequisite_subject_id = null,
    } = req.body;
    const con = await pool.getConnection();

    try {
      const [subjectExists] = await con.query(
        `SELECT * FROM subjects WHERE code = ? AND subject_id != ?`,
        [code, subject_id]
      );
      if (subjectExists.length > 0) {
        return res.status(400).json({
          status: 400,
          error: { msg: "Another subject with this code already exists!" },
        });
      }

      await con.query(
        `UPDATE subjects SET name = ?, code = ?, description = ?, learning_outcomes = ?, prerequisite_subject_id = ? WHERE subject_id = ?`,
        [
          name,
          code,
          description,
          learning_outcomes,
          prerequisite_subject_id,
          subject_id,
        ]
      );

      return res.status(200).json({
        status: 200,
        success: { msg: "Subject updated successfully" },
      });
    } catch (error) {
      console.error("Error:", error);
      return res.status(500).json({ error: { msg: "Internal server error" } });
    } finally {
      con.release();
    }
  } catch (error) {
    console.error("Database connection error:", error);
    return res.status(500).json({ error: { msg: "Internal server error" } });
  }
};
module.exports.getAllSubjects = async (req, res) => {
  try {
    const { searchKeyWord = "", pageNumber = 1, pageSize = 10 } = req.body;
    const offset = (pageNumber - 1) * pageSize;
    const con = await pool.getConnection();

    try {
      const [totalResult] = await con.query(
        `SELECT COUNT(*) AS total FROM subjects WHERE name LIKE ? AND is_deleted = FALSE`,
        [`%${searchKeyWord}%`]
      );

      const [subjects] = await con.query(
        `SELECT * FROM subjects WHERE name LIKE ? AND is_deleted = FALSE ORDER BY subject_id DESC LIMIT ? OFFSET ?`,
        [`%${searchKeyWord}%`, parseInt(pageSize), offset]
      );

      return res.status(200).json({
        status: 200,
        total: totalResult[0].total,
        pageSize,
        pageNumber,
        totalPages: Math.ceil(totalResult[0].total / pageSize),
        subjects,
      });
    } catch (error) {
      console.error("Error:", error);
      return res.status(500).json({ error: { msg: "Internal server error" } });
    } finally {
      con.release();
    }
  } catch (error) {
    console.error("Database connection error:", error);
    return res.status(500).json({ error: { msg: "Internal server error" } });
  }
};
module.exports.getSubjectById = async (req, res) => {
  try {
    const { subject_id } = req.body;
    const con = await pool.getConnection();

    try {
      const [result] = await con.query(
        `SELECT * FROM subjects WHERE subject_id = ? AND is_deleted = FALSE`,
        [subject_id]
      );

      return res.status(200).json({ status: 200, subject: result });
    } catch (error) {
      console.error("Error:", error);
      return res.status(500).json({ error: { msg: "Internal server error" } });
    } finally {
      con.release();
    }
  } catch (error) {
    console.error("Database connection error:", error);
    return res.status(500).json({ error: { msg: "Internal server error" } });
  }
};
module.exports.deleteSubject = async (req, res) => {
  try {
    const { ids } = req.body; // expecting array of subject_ids
    const con = await pool.getConnection();

    try {
      const [subjectExists] = await con.query(
        `SELECT * FROM subjects WHERE subject_id IN (?) AND is_deleted = FALSE`,
        [ids]
      );

      if (subjectExists.length === 0) {
        return res
          .status(404)
          .json({ status: 404, error: { msg: "Subjects not found" } });
      }

      const [deleteResult] = await con.query(
        `UPDATE subjects SET is_deleted = TRUE WHERE subject_id IN (?)`,
        [ids]
      );

      if (deleteResult.affectedRows > 0) {
        return res
          .status(200)
          .json({ status: 200, msg: "Subjects deleted successfully" });
      } else {
        return res.status(404).json({
          status: 404,
          error: { msg: "No matching records found to delete" },
        });
      }
    } catch (error) {
      console.error("Error:", error);
      return res.status(500).json({ error: { msg: "Internal server error" } });
    } finally {
      con.release();
    }
  } catch (error) {
    console.error("Database connection error:", error);
    return res.status(500).json({ error: { msg: "Internal server error" } });
  }
};
