const pool = require("../config/conn.js"); // Ensure correct database connection import

// ✅ Add or Update Student Thumb
module.exports.addOrUpdateStudentThumb = async (req, res) => {
  let connection;
  try {
    connection = await pool.getConnection();
    const {
      id,
      student_id,
      relation_id,
      name,
      impression_picture,
      iso_template,
    } = req.body;

    if (!impression_picture || !iso_template) {
      return res
        .status(400)
        .json({ error: "Impression picture or ISO Template missing" });
    }

    // Check if student_thumb already exists for the given id
    let existingRecord = [];
    if (id) {
      const checkQuery = `SELECT 1 FROM student_thumb WHERE id = ? LIMIT 1`;
      [existingRecord] = await connection.query(checkQuery, [id]);
    }

    if (existingRecord.length > 0) {
      // ✅ Update existing record
      const updateQuery = `
        UPDATE student_thumb 
        SET 
          relation_id = ?, 
          name = ?, 
          impression_picture = ?, 
          iso_template = ?, 
          updated_at = NOW()
        WHERE id = ?`;

      const updateValues = [
        relation_id,
        name,
        impression_picture,
        iso_template,
        id,
      ];
      await connection.query(updateQuery, updateValues);

      return res.status(200).json({
        status: 200,
        message: "Student thumb updated successfully!",
      });
    } else {
      // ❌ Insert new record
      const insertQuery = `
        INSERT INTO student_thumb 
          (student_id, relation_id, name, impression_picture, iso_template, created_at)
        VALUES (?, ?, ?, ?, ?, NOW())`;

      const insertValues = [
        student_id,
        relation_id,
        name,
        impression_picture,
        iso_template,
      ];
      await connection.query(insertQuery, insertValues);

      return res.status(201).json({
        status: 200,
        message: "Student thumb added successfully!",
      });
    }
  } catch (error) {
    console.error("Error in addOrUpdateStudentThumb function:", error);
    return res.status(500).json({ error: "Internal server error" });
  } finally {
    if (connection) connection.release();
  }
};

// ✅ Get Student Thumb by Student ID
module.exports.getStudentThumbById = async (req, res) => {
  let connection;
  try {
    connection = await pool.getConnection();
    const { id } = req.body;
    const query = `SELECT * FROM student_thumb WHERE id = ?`;
    const [results] = await connection.query(query, [id]);
    if (results.length > 0) {
      return res.status(200).json({
        message: "Student thumb retrieved successfully!",
        data: results[0],
        status: 200,
      });
    } else {
      return res.status(404).json({ message: "Student thumb not found." });
    }
  } catch (error) {
    console.error("Error in getStudentThumbById function:", error);
    return res.status(500).json({ error: "Internal server error" });
  } finally {
    if (connection) connection.release();
  }
};

// ✅ Get All Student Thumbs
module.exports.getAllStudentThumbs = async (req, res) => {
  try {
    const {
      searchKeyWord = "",
      student_id,
      pageNumber = 1,
      pageSize = 10,
    } = req.body;
    const offset = (pageNumber - 1) * pageSize;
    const con = await pool.getConnection();
    try {
      const [totalResult] = await con.query(
        `SELECT COUNT(*) AS total FROM student_thumb WHERE name LIKE ? AND student_id = ? AND deleted_at IS NULL`,
        [`%${searchKeyWord}%`, student_id]
      );

      const [thumbs] = await con.query(
        `SELECT * FROM student_thumb WHERE name LIKE ? AND student_id = ? AND deleted_at IS NULL ORDER BY id DESC LIMIT ? OFFSET ?`,
        [`%${searchKeyWord}%`, student_id, parseInt(pageSize), offset]
      );

      return res.status(200).json({
        status: 200,
        total: totalResult[0].total,
        pageSize,
        pageNumber,
        totalPages: Math.ceil(totalResult[0].total / pageSize),
        thumbs,
      });
    } catch (error) {
      console.error("Error:", error);
      return res.status(500).json({ error: { msg: "Internal server error" } });
    } finally {
      con.release();
    }
  } catch (error) {
    console.error("Database connection error:", error);
    return res.status(500).json({ error: { msg: "Internal server error" } });
  }
};

// ✅ Delete Student Thumb by ID
module.exports.deleteStudentThumb = async (req, res) => {
  let connection;
  try {
    connection = await pool.getConnection();
    const { student_id } = req.params;

    const deleteQuery = `DELETE FROM student_thumb WHERE student_id = ?`;
    const [result] = await connection.query(deleteQuery, [student_id]);

    if (result.affectedRows === 0) {
      return res.status(404).json({ message: "Student thumb not found." });
    }
    return res
      .status(200)
      .json({ message: "Student thumb deleted successfully!" });
  } catch (error) {
    console.error("Error in deleteStudentThumb function:", error);
    return res.status(500).json({ error: "Internal server error" });
  } finally {
    if (connection) connection.release();
  }
};

// soft delete student thumbs
module.exports.deleteThumbs = async (req, res) => {
  try {
    const { ids } = req.body;
    const con = await pool.getConnection();
    const [result] = await con.query(
      `UPDATE student_thumb SET deleted_at = NOW() WHERE id IN (?) AND deleted_at IS NULL`,
      [ids]
    );

    if (result.affectedRows === 0) {
      return res.status(404).json({
        success: false,
        error: "Thumbs not found or already deleted",
      });
    }
    res.status(200).json({ success: true, msg: "Thumbs deleted successfully" });
  } catch (error) {
    res.status(500).json({ success: false, error: error.message });
  }
};
