const pool = require("../config/db.js");
const bcrypt = require("bcrypt");
const jwt = require("jsonwebtoken");
const dotenv = require("dotenv");

dotenv.config();

module.exports.loginUser = async (req, res) => {
    let connection;
    try {
        const { email, password } = req.body;

        if (!email || !password) {
            return res.status(400).json({
                status: 400,
                error: { msg: "Invalid email or password" }
            });
        }

        connection = await pool.getConnection();

        // 1. Fetch user data along with their role name
        const userQuery = `
            SELECT u.*, r.name as role_name 
            FROM users u
            LEFT JOIN roles r ON u.role_id = r.id
            WHERE BINARY u.email = ? AND u.deleted_at IS NULL
        `;
        const [users] = await connection.query(userQuery, [email]);
        if (users.length === 0) {
            return res.status(401).json({ status: 401, message: "Invalid credentials" });
        }

        const user = users[0];

        let nodeHash = user.password.replace("$2y$", "$2b$");
        const isMatch = await bcrypt.compare(password, nodeHash);
        if (!isMatch) {
            return res.status(401).json({ status: 401, message: "Invalid credentials" });
        }

        // 2. Fetch permissions for the user's role
        let permissions = [];
        if (user.role_id) {
            const permissionQuery = `
                SELECT p.name FROM permissions p
                JOIN role_permissions rp ON p.id = rp.permission_id
                WHERE rp.role_id = ?
            `;
            const [permissionRows] = await connection.query(permissionQuery, [user.role_id]);
            permissions = permissionRows.map(p => p.name);
        }

        // 3. Generate JWT with user ID, role ID, and role name
        const token = jwt.sign(
            { 
                userId: user.id, 
                roleId: user.role_id,
                role: user.role_name
            },
            process.env.ACCESS_TOKEN_SECRET,
            { expiresIn: "30d" }
        );

        // 4. Prepare user data for response, including role and permissions
        const { password: _, email_verified_at, remember_token, ...userData } = user;
        userData.role = user.role_name; // Add role name
        userData.permissions = permissions; // Add permissions array
        return res.status(200).json({
            status: 200,
            data: { status: 200, user: userData, token, message: "Login successful!" }
        });

    } catch (error) {
        console.error("Internal server error:", error);
        return res.status(500).json({
            status: 500,
            error: { msg: "Internal server error" }
        });
    } finally {
        if (connection) connection.release();
    }
};

module.exports.addUser = async (req, res) => {
    let connection;
    try {
        const { first_name, last_name, email, password, role_id, student_id, teacher_id } = req.body;

        // 1. Basic validation
        if (!first_name || !last_name || !email || !password || !role_id) {
            return res.status(400).json({
                status: 400,
                error: { msg: "All fields (first_name, last_name, email, password, role_id) are required." }
            });
        }

        connection = await pool.getConnection();

        // 2. Validate role exists and get role name
        const [roleResult] = await connection.query(
            "SELECT name FROM roles WHERE id = ?",
            [role_id]
        );
        if (roleResult.length === 0) {
            return res.status(400).json({ status: 400, error: { msg: "Invalid role_id. Role not found." } });
        }

        const roleName = roleResult[0].name.toLowerCase();

        // 3. Check if user already exists by email
        const [existingUser] = await connection.query(
            "SELECT id FROM users WHERE email = ?",
            [email]
        );
        if (existingUser.length > 0) {
            return res.status(409).json({ status: 409, error: { msg: "A user with this email already exists." } });
        }

        // 4. Student role check
        if (roleName === "student") {
            if (!student_id) {
                return res.status(400).json({ status: 400, error: { msg: "student_id is required for student role." } });
            }

            const [studentCheck] = await connection.query(
                "SELECT user_id FROM students WHERE id = ?",
                [student_id]
            );
            if (studentCheck.length === 0) {
                return res.status(404).json({ status: 404, error: { msg: "Student not found." } });
            }

            if (studentCheck[0].user_id !== null) {
                return res.status(409).json({ status: 409, error: { msg: "This student is already linked to a user." } });
            }
        }

        // 5. Teacher role check
        if (roleName === "teacher") {
            if (!teacher_id) {
                return res.status(400).json({ status: 400, error: { msg: "teacher_id is required for teacher role." } });
            }

            const [teacherCheck] = await connection.query(
                "SELECT user_id FROM teachers WHERE id = ?",
                [teacher_id]
            );
            if (teacherCheck.length === 0) {
                return res.status(404).json({ status: 404, error: { msg: "Teacher not found." } });
            }

            if (teacherCheck[0].user_id !== null) {
                return res.status(409).json({ status: 409, error: { msg: "This teacher is already linked to a user." } });
            }
        }

        // 6. Hash the password
        const salt = await bcrypt.genSalt(10);
        const hashedPassword = await bcrypt.hash(password, salt);

        // 7. Insert the new user
        const [insertResult] = await connection.query(
            `INSERT INTO users 
                (first_name, last_name, email, password, role_id, created_at, updated_at)
             VALUES (?, ?, ?, ?, ?, NOW(), NOW())`,
            [first_name, last_name, email, hashedPassword, role_id]
        );

        const newUserId = insertResult.insertId;

        // 8. Update student or teacher to link user_id
        if (roleName === "student") {
            await connection.query(
                "UPDATE students SET user_id = ? WHERE id = ?",
                [newUserId, student_id]
            );
        }

        if (roleName === "teacher") {
            await connection.query(
                "UPDATE teachers SET user_id = ? WHERE id = ?",
                [newUserId, teacher_id]
            );
        }

        return res.status(200).json({
            status: 200,
            success: {
                msg: "User created and linked successfully.",
                userId: newUserId
            }
        });

    } catch (error) {
        console.error("Error creating user:", error);
        return res.status(500).json({ status: 500, error: { msg: "Internal server error" } });
    } finally {
        if (connection) connection.release();
    }
};


module.exports.getAllUsers = async (req, res) => {
    let connection;
    try {
        const {
            searchKeyWord = "",
            role_id, // New filter
            pageNumber = 1,
            pageSize = 10
        } = req.body;

        const offset = (parseInt(pageNumber, 10) - 1) * parseInt(pageSize, 10);
        connection = await pool.getConnection();

        let whereClauses = ["u.deleted_at IS NULL"];
        const params = [];

        if (searchKeyWord) {
            const searchPattern = `%${searchKeyWord}%`;
            whereClauses.push('(u.first_name LIKE ? OR u.last_name LIKE ? OR u.email LIKE ?)');
            params.push(searchPattern, searchPattern, searchPattern);
        }

        if (role_id) {
            whereClauses.push('u.role_id = ?');
            params.push(role_id);
        }

        const whereString = `WHERE ${whereClauses.join(" AND ")}`;

        // Get total count
        const [totalResult] = await connection.query(
            `SELECT COUNT(u.id) as total FROM users u ${whereString}`,
            params
        );
        const totalRecords = totalResult[0].total;

        // Get paginated user data with role name
        const queryParams = [...params, parseInt(pageSize, 10), offset];
        const [users] = await connection.query(
            `SELECT u.id, u.first_name, u.last_name, u.email, u.role_id, r.name as role_name, u.created_at, u.updated_at 
             FROM users u
             LEFT JOIN roles r ON u.role_id = r.id
             ${whereString} 
             ORDER BY u.created_at DESC 
             LIMIT ? OFFSET ?`,
            queryParams
        );

        return res.status(200).json({
            status: 200,
            total: totalRecords,
            pageSize: parseInt(pageSize, 10),
            pageNumber: parseInt(pageNumber, 10),
            totalPages: Math.ceil(totalRecords / parseInt(pageSize, 10)),
            users: users
        });

    } catch (error) {
        console.error("Error fetching all users:", error);
        return res.status(500).json({ status: 500, error: { msg: "Internal server error" } });
    } finally {
        if (connection) connection.release();
    }
};

module.exports.getUserById = async (req, res) => {
    let connection;
    try {
        const { user_id } = req.body;

        if(!user_id) {
             return res.status(400).json({ error: { msg: "User ID is required" } });
        }
        
        connection = await pool.getConnection();
      
        // Get user details and role name
        const [userRows] = await connection.query(
          `SELECT u.id, u.first_name, u.last_name, u.email, u.role_id, r.name as role_name, u.created_at, u.updated_at 
           FROM users u
           LEFT JOIN roles r ON u.role_id = r.id
           WHERE u.id = ? AND u.deleted_at IS NULL`,
          [user_id]
        );

        if (userRows.length === 0) {
            return res.status(404).json({ error: { msg: "User not found" } });
        }

        const user = userRows[0];

        // Get user permissions
        const [permissionRows] = await connection.query(
            `SELECT p.id, p.name FROM permissions p JOIN role_permissions rp ON p.id = rp.permission_id WHERE rp.role_id = ?`,
            [user.role_id]
        );
        user.permissions = permissionRows;

        return res.status(200).json({ status: 200, user: user });

    } catch (error) {
        console.error("Error fetching user by ID:", error);
        return res.status(500).json({ error: { msg: "Internal server error" } });
    } finally {
        if (connection) connection.release();
    }
  };

module.exports.updateUser = async (req, res) => {
    let connection;
    try {
        const { user_id, first_name, last_name, email, password, role_id } = req.body;

        if (!user_id) {
            return res.status(400).json({ status: 400, error: { msg: "User ID is required." } });
        }

        connection = await pool.getConnection();
        const updateFields = [];
        const updateParams = [];

        if (first_name) { updateFields.push("first_name = ?"); updateParams.push(first_name); }
        if (last_name) { updateFields.push("last_name = ?"); updateParams.push(last_name); }
        if (role_id) { updateFields.push("role_id = ?"); updateParams.push(role_id); }

        if (email) {
            const [existingUser] = await connection.query("SELECT id FROM users WHERE email = ? AND id != ?", [email, user_id]);
            if (existingUser.length > 0) {
                return res.status(409).json({ status: 409, error: { msg: "This email is already in use by another account." } });
            }
            updateFields.push("email = ?");
            updateParams.push(email);
        }
        if (password) {
            const salt = await bcrypt.genSalt(10);
            const hashedPassword = await bcrypt.hash(password, salt);
            updateFields.push("password = ?");
            updateParams.push(hashedPassword);
        }

        if (updateFields.length === 0) {
            return res.status(400).json({ status: 400, error: { msg: "No fields to update were provided." } });
        }
        
        updateParams.push(user_id);
        const [result] = await connection.query(
            `UPDATE users SET ${updateFields.join(", ")} WHERE id = ? AND deleted_at IS NULL`,
            updateParams
        );

        if (result.affectedRows === 0) {
            return res.status(404).json({ status: 404, error: { msg: "User not found or no changes were made." } });
        }

        return res.status(200).json({ status: 200, success: { msg: "User updated successfully." } });

    } catch (error) {
        console.error("Error updating user:", error);
        if (error.code === 'ER_NO_REFERENCED_ROW_2') {
             return res.status(400).json({ status: 400, error: { msg: "Invalid role_id. The specified role does not exist." } });
        }
        return res.status(500).json({ status: 500, error: { msg: "Internal server error" } });
    } finally {
        if (connection) connection.release();
    }
};

module.exports.deleteUser = async (req, res) => {
    // This function remains largely the same for soft-delete.
    // Add dependency checks here if hard deletes or more complex logic is needed.
    let connection;
    try {
      const { ids } = req.body; 
      if (!Array.isArray(ids) || ids.length === 0) {
        return res.status(400).json({ status: 400, error: { msg: "User IDs are required as a non-empty array." } });
      }
  
      connection = await pool.getConnection();
      
      const [usersExist] = await connection.query(
        `SELECT id FROM users WHERE id IN (?) AND deleted_at IS NULL`,
        [ids]
      );
      if (usersExist.length === 0) {
        return res.status(404).json({ status: 404, error: { msg: "Users not found or already deleted." } });
      }

      const [deleteResult] = await connection.query(
        `UPDATE users SET deleted_at = NOW() WHERE id IN (?)`,
        [ids]
      );

      if (deleteResult.affectedRows > 0) {
        return res.status(200).json({ status: 200, msg: "Users deleted successfully." });
      } else {
        return res.status(404).json({ status: 404, error: { msg: "No matching users found to delete." } });
      }
    } catch (error) {
        console.error("Error deleting users:", error);
        return res.status(500).json({ status: 500, error: { msg: "Internal server error." } });
    } finally {
        if (connection) connection.release();
    }
};
module.exports.getAllRoles = async (req,res)=>{

    let connection;
    try{
        connection = await pool.getConnection();
        const [roles] = await connection.query("SELECT id, name FROM roles ORDER BY name ASC");
        return res.status(200).json({ status: 200, roles: roles });
    }catch(error){
        console.error("Error fetching roles:", error);
        return res.status(500).json({ status: 500, error: { msg: "Internal server error." } });
    }
}