const jwt = require('jsonwebtoken');
const pool = require("../config/db.js");
const dotenv = require("dotenv");

dotenv.config();


const authenticateToken = (req, res, next) => {
    const authHeader = req.headers['authorization'];
    const token = authHeader && authHeader.split(' ')[1]; 

    if (token == null) {
        return res.status(401).json({ status: 401, error: { msg: "Unauthorized: No token provided" } });
    }

    jwt.verify(token, process.env.ACCESS_TOKEN_SECRET, (err, user) => {
        if (err) {
            return res.status(403).json({ status: 403, error: { msg: "Forbidden: Invalid or expired token" } });
        }
        req.user = user; 
        next(); 
    });
};



const hasPermission = (requiredPermission) => {
    return async (req, res, next) => {
        if (!req.user || !req.user.roleId) {
            return res.status(403).json({ status: 403, error: { msg: "Forbidden: No role associated with user" } });
        }

        let connection;
        try {
            connection = await pool.getConnection();

            const permissionQuery = `
                SELECT p.name 
                FROM permissions p
                JOIN role_permissions rp ON p.id = rp.permission_id
                WHERE rp.role_id = ?
            `;
            const [permissionRows] = await connection.query(permissionQuery, [req.user.roleId]);
            const userPermissions = permissionRows.map(p => p.name);

            if (userPermissions.includes(requiredPermission)) {
                next(); 
            } else {
                return res.status(403).json({ status: 403, error: { msg: "Forbidden: You do not have permission to perform this action." } });
            }
        } catch (error) {
            console.error("Permission check error:", error);
            return res.status(500).json({ status: 500, error: { msg: "Internal server error during permission check." } });
        } finally {
            if (connection) connection.release();
        }
    };
};


module.exports = {
    authenticateToken,
    hasPermission
};